package com.zell_mbc.publicartexplorer.helpers

import android.content.Context
import android.net.ConnectivityManager
import android.net.Network
import android.net.NetworkCapabilities
import android.net.NetworkRequest
import android.widget.Toast
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow

class NetworkStatusTracker(context: Context) {
    private val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

    // Using SharedFlow for better handling of multiple consumers, though StateFlow is fine.
    // Sticking to StateFlow as in your original code.
    private val _isOnline = MutableStateFlow(false)
    val isOnline: StateFlow<Boolean> = _isOnline

    private val networkCallback = object : ConnectivityManager.NetworkCallback() {

        // --- 1. Available Callback: Internet is definitely present ---
        override fun onAvailable(network: Network) {
            // When a network with the requested capabilities becomes available.
            // This is a reliable signal that internet access is present.
            _isOnline.value = true
            //Toast.makeText(context, "onAvailable: _isOnline=${_isOnline.value}", Toast.LENGTH_LONG).show()
        }

        // --- 2. Lost Callback: A previously available network is lost ---
        override fun onLost(network: Network) {
            // A specific network has been lost. We must immediately check if
            // *any* other network capable of internet is currently active.
            checkCurrentNetworkStatus()
            //Toast.makeText(context, "onLost: _isOnline=${_isOnline.value}", Toast.LENGTH_LONG).show()
        }

        // --- 3. Capability Change: Use this for finer-grained changes ---
        override fun onCapabilitiesChanged(network: Network, networkCapabilities: NetworkCapabilities) {
            // This is triggered when a network is connected but its capabilities change (e.g.,
            // it's an internet connection, but captive portal is detected, or the internet
            // capability is reported lost). We re-run the check.
            checkCurrentNetworkStatus()
            //Toast.makeText(context, "onCabilitiesChanged: _isOnline=${_isOnline.value}", Toast.LENGTH_LONG).show()
        }

        // --- 4. Network Blocked: Useful if a network loses internet access without being "lost" ---
        override fun onBlockedStatusChanged(network: Network, blocked: Boolean) {
            checkCurrentNetworkStatus()
            //Toast.makeText(context, "onBlocked: _isOnline=${_isOnline.value}", Toast.LENGTH_LONG).show()
        }
    }

    // --- Helper function for robust checking ---
    private fun checkCurrentNetworkStatus() {
        // Find the currently active default network
        val network = connectivityManager.activeNetwork

        // Get the capabilities for that network
        val capabilities = connectivityManager.getNetworkCapabilities(network)

        // Determine if the active network has the INTERNET capability
        val hasInternet = capabilities?.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET) == true

        // Update the status
        _isOnline.value = hasInternet
    }

    fun startListening() {
        val request = NetworkRequest.Builder()
            .addCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)
            .build()

        // Register the callback to start monitoring
        connectivityManager.registerNetworkCallback(request, networkCallback)

        // Perform the initial check immediately
        checkCurrentNetworkStatus()
    }

    fun stopListening() {
        connectivityManager.unregisterNetworkCallback(networkCallback)
    }
}
/*
// This app needs internet to function,
// This class monitors if Internet is available so we can let users know instead of showing timeouts and similar technical warnings.
class NetworkStatusTracker(context: Context) {
    private val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

    private val _isOnline = MutableStateFlow(false)
    val isOnline: StateFlow<Boolean> = _isOnline

    private val networkCallback = object : ConnectivityManager.NetworkCallback() {
        override fun onAvailable(network: Network) {
            _isOnline.value = true
            //Toast.makeText(context, "onAvailable: _isOnline=${_isOnline.value}" , Toast.LENGTH_LONG).show()
        }

        override fun onLost(network: Network) {
            val current = connectivityManager.activeNetwork
            val caps = connectivityManager.getNetworkCapabilities(current)

            val stillOnline = caps?.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET) == true
            _isOnline.value = true // stillOnline
            //Toast.makeText(context, "onLost: _isOnline=${_isOnline.value}", Toast.LENGTH_LONG).show()
        }
    }

    fun startListening() {
        val request = NetworkRequest.Builder()
            .addCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)
            .build()
        connectivityManager.registerNetworkCallback(request, networkCallback)

        // initial check
        val network = connectivityManager.activeNetwork
        val capabilities = connectivityManager.getNetworkCapabilities(network)
        _isOnline.value = capabilities?.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET) == true
    }

    fun stopListening() {
        connectivityManager.unregisterNetworkCallback(networkCallback)
    }
}

 */