package com.zell_mbc.publicartexplorer.detailview

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.LocationOff
import androidx.compose.material.icons.filled.LocationOn
import androidx.compose.material3.FilterChip
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import com.zell_mbc.publicartexplorer.data.ViewModel

@Composable
fun WikiCategorySelector(
    viewModel: ViewModel,
    suggestedCategories: List<String>,
    initialSelection: List<String> = emptyList(),
    hasGps: Boolean,
    onSelectionChanged: (List<String>) -> Unit
) {
    val context = LocalContext.current
    var selectedCategories by remember { mutableStateOf(initialSelection) }
    var customCategory by remember { mutableStateOf("") }

    Column(modifier = Modifier.fillMaxWidth().padding(8.dp).verticalScroll(rememberScrollState()) ) {
        Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.SpaceBetween, verticalAlignment = Alignment.CenterVertically) {
            // Suggested categories
            Text("Suggested", style = MaterialTheme.typography.titleMedium)

            Icon(
                imageVector = if (hasGps) Icons.Default.LocationOn else Icons.Default.LocationOff,
                contentDescription = if (hasGps) "GPS data found" else "No GPS data",
                tint = if (hasGps) Color.Green else Color.Gray,
                modifier = Modifier.size(24.dp).clickable {
                    viewModel.showMessage(if (hasGps) "Image contains GPS data" else "Image contains no GPS data")
                    //Toast.makeText(context, if (hasGps) "Image contains GPS data" else "Image contains no GPS data", Toast.LENGTH_SHORT).show()
                }
            )
        }

        FlowRow(modifier = Modifier.padding(top = 8.dp),) {
            suggestedCategories.forEach { category ->
                val isSelected = category in selectedCategories
                FilterChip(
                    selected = isSelected,
                    onClick = {
                        selectedCategories = if (isSelected) {
                            selectedCategories - category
                        } else { selectedCategories + category }
                        onSelectionChanged(selectedCategories)
                    },
                    label = { Text(category, style = MaterialTheme.typography.bodySmall) }
                )
            }
        }

        Spacer(Modifier.height(12.dp))

        // Manual category input
        Row(
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier.fillMaxWidth()
        ) {
            TextField(
                value = customCategory,
                onValueChange = { customCategory = it },
                placeholder = { Text("Add or edit category") },
                singleLine = true,
                modifier = Modifier.weight(1f)
            )
            IconButton(
                onClick = {
                    val trimmed = customCategory.trim()
                    if (trimmed.isNotEmpty()) {
                        if (!selectedCategories.contains(trimmed)) {
                            selectedCategories = selectedCategories + trimmed
                            onSelectionChanged(selectedCategories)
                        }
                        customCategory = "" // clear after add
                    }
                }
            ) {
                Icon(Icons.Default.Add, contentDescription = "Add category")
            }
        }

        Spacer(Modifier.height(12.dp))

        if (selectedCategories.isNotEmpty()) {
            Text("Selected", style = MaterialTheme.typography.titleMedium)
            FlowRow(
                modifier = Modifier.padding(top = 8.dp),
                horizontalArrangement = Arrangement.spacedBy(4.dp),
                verticalArrangement = Arrangement.spacedBy(4.dp)
            ) {
                selectedCategories.forEach { category ->
                    FilterChip(
                        selected = false,
                        onClick = {
                            // put category back into text field for editing
                            customCategory = category
                            selectedCategories = selectedCategories - category
                            onSelectionChanged(selectedCategories)
                        },
                        label = { Text(category, style = MaterialTheme.typography.bodySmall) }
                    )
                }
            }
        }
    }
}