package com.zell_mbc.publicartexplorer.detailview

import android.content.Intent
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.text.selection.SelectionContainer
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import com.zell_mbc.publicartexplorer.Artwork
import com.zell_mbc.publicartexplorer.ClickableEmail
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.capitalize
import com.zell_mbc.publicartexplorer.createClickableUrl
import com.zell_mbc.publicartexplorer.data.ViewModel
import com.zell_mbc.publicartexplorer.getLocalValue
import kotlin.text.contains
import kotlin.text.isNotEmpty
import androidx.core.net.toUri
import com.zell_mbc.publicartexplorer.ClickableUrlText
import com.zell_mbc.publicartexplorer.helpers.createWikipediaUrl
import com.zell_mbc.publicartexplorer.isPhotoAlbum

@Composable
fun ShowDetails(viewModel:ViewModel, selectedArtwork: Artwork) {
    val context = LocalContext.current
    //var firstLoaded by remember { mutableStateOf<String?>(null) } // stores which source got a picture first
    SelectionContainer {
    Column(modifier = Modifier.fillMaxWidth().verticalScroll(rememberScrollState())) {

        // Show all images
        if (viewModel.imageItems.size> 0) ImageCarousel(viewModel.imageItems)

        // Need to check here again because of Google Photos, Flickr and the likes
        val imageUrl = selectedArtwork.tags["image"] ?: ""
        if (imageUrl.isNotEmpty()) {
            val photoService = isPhotoAlbum(imageUrl)
            // Example: Istanbul
            if (photoService != null) { // No way to determine stable image name from Google Photos and the likes: Show as website link
                ClickableUrlText(
                    imageUrl,
                    urlText = photoService,
                    textBefore = (stringResource(R.string.imageAlbum) + ": "),
                    textColor = MaterialTheme.colorScheme.onSurface
                )
            }
        }

        // **********************************
        // Text part
        // **********************************
        //if (!firstLoaded.isNullOrEmpty()) Text("") // Blank line if an image is shown

        // Artwork name and link
        var s = selectedArtwork.tags["name"] ?: ""
        if (s.isNotEmpty()) Text(
            context.getString(R.string.tag_name) + ": $s",
            color = MaterialTheme.colorScheme.onSurface
        )

        s = createWikipediaUrl(selectedArtwork.tags["subject:wikipedia"] ?: "")
        if (s.isNotEmpty()) {
            val linkText = stringResource(R.string.artwork) + " Wikipedia"
            Text(createClickableUrl(s, linkText), color = MaterialTheme.colorScheme.onSurface)
            Text("")
        }

        //s = selectedArtwork.tags["description"] ?: ""
        s = getLocalValue("description", selectedArtwork.tags, context)
        if (s.isNotEmpty()) {
            Text(
                context.getString(R.string.tag_description) + ": $s",
                color = MaterialTheme.colorScheme.onSurface
            )
            Text("")
        }

        val nodeType = selectedArtwork.tags["tourism"] ?: ""
        // Museum specific data
        if (nodeType.contains("museum")) {
            s = selectedArtwork.tags["opening_hours"] ?: ""
            if (s.isNotEmpty()) Text(s, color = MaterialTheme.colorScheme.onSurface)
            s = selectedArtwork.tags["phone"] ?: ""
            if (s.isNotEmpty()) Text(s, color = MaterialTheme.colorScheme.onSurface)
            s = selectedArtwork.tags["email"] ?: ""
            if (s.isNotEmpty()) ClickableEmail(s)

            s = selectedArtwork.tags["wikipedia"] ?: ""
            if (s.isNotEmpty()) {
                val tmp = createWikipediaUrl(s)
                if (tmp.isNotEmpty()) {
                    Text(createClickableUrl(tmp), color = MaterialTheme.colorScheme.onSurface)
                }
            }
        }
        // All the other ones
        else {
            // Artist
            s = selectedArtwork.tags["artist_name"] ?: ""
            if (s.isNotEmpty()) Text(
                context.getString(R.string.tag_artist) + ": $s",
                color = MaterialTheme.colorScheme.onSurface
            )

            s = selectedArtwork.tags["artist:wikipedia"] ?: ""
            ArtistWikipedia(s)

            s = selectedArtwork.tags["artist:wikidata"] ?: ""
            ArtistWikidata(s)

            // Artwork subject
            val subject = selectedArtwork.tags["artwork_subject"] ?: ""
            s = if (subject.isNotEmpty()) {
                val resId = context.resources.getIdentifier(subject, "string", context.packageName)
                if (resId != 0) context.getString(resId) else subject
            } else ""
            if (s.isNotEmpty()) {
                s =
                    capitalize(s) //.replaceFirstChar { if (it.isLowerCase()) it.titlecase() else it.toString() }
                Text(
                    context.getString(R.string.tag_subject) + ": $s",
                    color = MaterialTheme.colorScheme.onSurface
                )
            }

            // Material
            var tag = selectedArtwork.tags["material"] ?: ""
            s = if (tag.isNotEmpty()) {
                val resId = context.resources.getIdentifier(tag, "string", context.packageName)
                if (resId != 0) context.getString(resId) else tag
            } else ""
            if (s.isNotEmpty()) {
                s = capitalize(s)
                // Can we be more specific?
                tag = selectedArtwork.tags["stone"] ?: ""
                if (tag.isNotEmpty()) s = "$s, $tag"

                Text(
                    context.getString(R.string.tag_material) + ": $s",
                    color = MaterialTheme.colorScheme.onSurface
                )
            }

            s = selectedArtwork.tags["start_date"] ?: ""
            if (s.isNotEmpty()) Text(
                context.getString(R.string.tag_created) + ": $s",
                color = MaterialTheme.colorScheme.onSurface
            )
        }

        // Website
        s = selectedArtwork.tags["website"] ?: selectedArtwork.tags["contact:website"] ?: ""
        if (s.isNotEmpty()) {
            Text(
                createClickableUrl(s, stringResource(R.string.website)),
                overflow = TextOverflow.Clip
            )
            Text("")
        }

        s = selectedArtwork.tags["wheelchair"] ?: ""
        if (s.isNotEmpty()) {
            val resId = context.resources.getIdentifier(s, "string", context.packageName)
            if (resId != 0) s = context.getString(resId)
            Text(
                context.getString(R.string.tag_wheelchair) + ": $s",
                color = MaterialTheme.colorScheme.onSurface
            )
        }

        // If this node is part of a relation show the relation specific data now
        ShowRelationsData(viewModel, selectedArtwork.id)
        Text("")
        HorizontalDivider(thickness = 1.dp, color = MaterialTheme.colorScheme.primary)
        Text("")

        Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.End) {
            OutlinedButton(
                onClick = {
                    val label = selectedArtwork.tags["name"] ?: context.getString(R.string.artwork)
                    val intentUri =
                        "geo:${selectedArtwork.lat},${selectedArtwork.lon}?q=${selectedArtwork.lat},${selectedArtwork.lon}($label)"
                    val intent = Intent(Intent.ACTION_VIEW, intentUri.toUri())
                    context.startActivity(intent)
                }
            ) {
                Text(stringResource(R.string.sendCoordinates), color = MaterialTheme.colorScheme.primary)
            }
        }
    }
    }
}

// Check if Image or category
@Composable
fun ArtistWikipedia(s: String) {
    val linkText = stringResource(R.string.tag_artist) + " Wikipedia"
    if (s.isNotEmpty())
        when {
            s.contains("http") -> {  // A proper url
                Text(createClickableUrl(s, linkText))
                Text("")
            }
            s.contains(":") -> { // Probably Wikipedia notation 'en:Peter Randal-Page or en:Peter_Randal-Page'
                val tmp = createWikipediaUrl(s)
                if (tmp.isNotEmpty()) {
                    Text(createClickableUrl(tmp, linkText))
                    Text("")
                }
            }
        }
}

// artist:wikipedia tag
@Composable
fun ArtistWikidata(s: String) {
    val linkText = stringResource(R.string.tag_artist) + " Wikidata"
    if (s.isNotEmpty())
        when {
            s.contains("http") -> {  // A proper url
                Text(createClickableUrl(s, linkText), color = MaterialTheme.colorScheme.onSurface)
                Text("")
            }
            s.contains(":") -> { // Probably Wikipedia notation 'en:Peter Randal-Page'
                val tmp = createWikipediaUrl(s)
                if (tmp.isNotEmpty()) {
                    Text(createClickableUrl(tmp, linkText), color = MaterialTheme.colorScheme.onSurface)
                    Text("")
                }
            }
            s.startsWith("Q") -> {
                val tmp = "https://www.wikidata.org/wiki/$s"
                Text(createClickableUrl(tmp, linkText), color = MaterialTheme.colorScheme.onSurface)
                Text("")
            }
        }
}
