package com.zell_mbc.publicartexplorer.detailview

import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.rememberPagerState
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.ChevronLeft
import androidx.compose.material.icons.filled.ChevronRight
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import coil3.compose.SubcomposeAsyncImage
import com.zell_mbc.publicartexplorer.CoilImageLoader
import com.zell_mbc.publicartexplorer.R
import kotlinx.coroutines.launch

data class ImageItem(
    val url: String,
    val provider: String,
    val attribution: String = ""
)

@Composable
fun ImageCarousel(imageItems: MutableList<ImageItem>) {
    var showFullscreenImage by remember { mutableStateOf(false) }
    var fullscreenPage by remember { mutableStateOf(0) }

    if (imageItems.isEmpty()) {
        Box(modifier = Modifier.fillMaxWidth(), contentAlignment = Alignment.Center) {
            Text(stringResource(R.string.unableToDownloadImage), color = Color.Red)
        }
        return
    }

    val pagerState = rememberPagerState(initialPage = 0, pageCount = { imageItems.size })

    Column {
        Box(modifier = Modifier.fillMaxWidth()
            //.height(350.dp) // adjust total height for image + text
            .clickable {
                fullscreenPage = pagerState.currentPage
                showFullscreenImage = true
            }) {
            val scope = rememberCoroutineScope()

            HorizontalPager(
                state = pagerState,
                modifier = Modifier.fillMaxSize()
            ) { page ->
                val item = imageItems[page]

                Column(
                    horizontalAlignment = Alignment.CenterHorizontally,
                    modifier = Modifier.fillMaxSize()
                ) {
                    SubcomposeAsyncImage(
                        model = item.url,
                        contentDescription = null,
                        imageLoader = CoilImageLoader.imageLoader,
                        modifier = Modifier
                            .fillMaxWidth(),
                        loading = { CircularProgressIndicator(modifier = Modifier.align(Alignment.Center)) },
                        error = {
                            // Remove this item if there are more items left
                            if (imageItems.size > 1) {
                                    val currentIndex = pagerState.currentPage
                                    imageItems.removeAt(currentIndex.coerceIn(imageItems.indices))
                            } else {
                                // Last item, just show the error text
                                Text(stringResource(R.string.unableToDownloadImage), color = Color.Red, modifier = Modifier.align(Alignment.Center))
                            }
                        }
                    )

                    // Attribution below the image
                    Text(
                        text = "© " + if (item.attribution.isEmpty()) item.provider else item.attribution,
                        fontSize = 12.sp,
                        fontStyle = FontStyle.Italic,
                        textAlign = TextAlign.Center,
                        modifier = Modifier.padding(top = 4.dp)
                    )
                }
            }

            // Left arrow
            if (pagerState.currentPage > 0) {
                IconButton(
                    onClick = {
                        scope.launch {
                            pagerState.animateScrollToPage(pagerState.currentPage - 1)
                        }
                              },
                    modifier = Modifier
                        .align(Alignment.CenterStart)
                        .size(40.dp)
                        .background(Color.Black.copy(alpha = 0.4f), shape = CircleShape)
                ) {
                    Icon(
                        imageVector = Icons.Default.ChevronLeft,
                        contentDescription = "Previous image",
                        tint = Color.White
                    )
                }
            }

            // Right arrow
            if (pagerState.currentPage < pagerState.pageCount - 1) {
                IconButton(
                    onClick = {
                        scope.launch { pagerState.animateScrollToPage(pagerState.currentPage + 1) }
                              },
                    modifier = Modifier
                        .align(Alignment.CenterEnd)
                        .size(40.dp)
                        .background(Color.Black.copy(alpha = 0.4f), shape = CircleShape)
                ) {
                    Icon(
                        imageVector = Icons.Default.ChevronRight,
                        contentDescription = "Next image",
                        tint = Color.White
                    )
                }
            }
        }

        Spacer(modifier = Modifier.height(4.dp))
    }

    // Fullscreen dialog with swipe + zoom
    if (showFullscreenImage) {
        Dialog(
            onDismissRequest = { showFullscreenImage = false },
            properties = DialogProperties(usePlatformDefaultWidth = false)
        ) {
            val fullscreenPagerState = rememberPagerState(
                initialPage = fullscreenPage,
                pageCount = { imageItems.size }
            )

            HorizontalPager(
                state = fullscreenPagerState,
                modifier = Modifier.fillMaxSize()
            ) { page ->
                ZoomableImage(
                    imageUrl = imageItems[page].url,
                    onClose = { showFullscreenImage = false }
                )
            }
        }
    }
}
