import android.content.Context
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyPermanentlyInvalidatedException
import android.security.keystore.KeyProperties
import android.util.Base64
import androidx.datastore.preferences.core.*
import androidx.datastore.preferences.preferencesDataStore
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.launch
import javax.crypto.Cipher
import javax.crypto.KeyGenerator
import javax.crypto.SecretKey
import javax.crypto.spec.GCMParameterSpec
import kotlin.getValue

// This is so much boilerplate code just for settings, unaceptable…not going to use this.
// ---------------- Plain DataStore ----------------
private val Context.dataStore by preferencesDataStore("settings")
class PreferencesManager(private val context: Context) {

    // ---------------- Generic Flows ----------------

    fun getStringFlow(key: String, default: String = ""): Flow<String> {
        val prefKey = stringPreferencesKey(key)
        return context.dataStore.data.map { it[prefKey] ?: default }
    }

    fun getBooleanFlow(key: String, default: Boolean = false): Flow<Boolean> {
        val prefKey = booleanPreferencesKey(key)
        return context.dataStore.data.map { it[prefKey] ?: default }
    }

    fun getIntFlow(key: String, default: Int = 0): Flow<Int> {
        val prefKey = intPreferencesKey(key)
        return context.dataStore.data.map { it[prefKey] ?: default }
    }

    fun getLongFlow(key: String, default: Long = 0L): Flow<Long> {
        val prefKey = longPreferencesKey(key)
        return context.dataStore.data.map { it[prefKey] ?: default }
    }

    fun getFloatFlow(key: String, default: Float = 0f): Flow<Float> {
        val prefKey = floatPreferencesKey(key)
        return context.dataStore.data.map { it[prefKey] ?: default }
    }

    // -------------------- Suspend getters for non-Compose --------------------

    suspend fun getBoolean(key: String, default: Boolean = false): Boolean =
        context.dataStore.data.map { it[booleanPreferencesKey(key)] ?: default }.first()

    suspend fun getInt(key: String, default: Int = 0): Int =
        context.dataStore.data.map { it[intPreferencesKey(key)] ?: default }.first()

    suspend fun getString(key: String, default: String = ""): String =
        context.dataStore.data.map { it[stringPreferencesKey(key)] ?: default }.first()

    // ---------------- Generic setters ----------------

    fun setString(key: String, value: String, scope: CoroutineScope) {
        scope.launch { setString(key, value) }
    }

    suspend fun setString(key: String, value: String) {
        val prefKey = stringPreferencesKey(key)
        context.dataStore.edit { it[prefKey] = value }
    }

    fun setBoolean(key: String, value: Boolean, scope: CoroutineScope) {
        scope.launch { setBoolean(key, value) }
    }

    suspend fun setBoolean(key: String, value: Boolean) {
        val prefKey = booleanPreferencesKey(key)
        context.dataStore.edit { it[prefKey] = value }
    }

    fun setInt(key: String, value: Int, scope: CoroutineScope) {
        scope.launch { setInt(key, value) }
    }

    suspend fun setInt(key: String, value: Int) {
        val prefKey = intPreferencesKey(key)
        context.dataStore.edit { it[prefKey] = value }
    }

    fun setLong(key: String, value: Long, scope: CoroutineScope) {
        scope.launch { setLong(key, value) }
    }

    suspend fun setLong(key: String, value: Long) {
        val prefKey = longPreferencesKey(key)
        context.dataStore.edit { it[prefKey] = value }
    }

    fun setFloat(key: String, value: Float, scope: CoroutineScope) {
        scope.launch { setFloat(key, value) }
    }

    suspend fun setFloat(key: String, value: Float) {
        val prefKey = floatPreferencesKey(key)
        context.dataStore.edit { it[prefKey] = value }
    }

    // ---------------- Generic remover ----------------

    suspend fun remove(key: String) {
        context.dataStore.edit { prefs ->
            prefs.remove(stringPreferencesKey(key))
            prefs.remove(booleanPreferencesKey(key))
            prefs.remove(intPreferencesKey(key))
            prefs.remove(longPreferencesKey(key))
            prefs.remove(floatPreferencesKey(key))
        }
    }
}
/* Datastore preferences / Use for settings which drive the UI
class PreferencesManager(private val context: Context) {

    // Generic plain Flow
    fun flow(key: String, default: String = ""): Flow<String> {
        val prefKey = stringPreferencesKey(key)
        return context.dataStore.data.map { prefs ->
            prefs[prefKey] ?: default
        }
    }

    // Generic plain setter
    fun set(key: String, value: String, scope: CoroutineScope) {
        scope.launch { set(key, value)}
    }

    suspend operator fun set(key: String, value: String) {
        val prefKey = stringPreferencesKey(key)
        context.dataStore.edit { prefs ->
            prefs[prefKey] = value
        }
    }

    // Generic plain remover
    suspend fun remove(key: String) {
        val prefKey = stringPreferencesKey(key)
        context.dataStore.edit { prefs ->
            prefs.remove(prefKey)
        }
    }

    suspend fun setEncrypted(key: String, value: String) {
        val encryptedValue = encrypt(value)
        set(key, encryptedValue) // store in DataStore
    }

    fun getEncryptedFlow(key: String, default: String = ""): Flow<String> {
        return flow(key, default).map { encrypted ->
            if (encrypted.isEmpty()) "" else decrypt(encrypted)
        }
    }

    // ---------------- Encryption helpers ----------------
    private val masterKey = generateMasterKey()

    // ---------- Encrypted DataStore Helpers ----------
    private val ALGORITHM = "AES/GCM/NoPadding"
    private val GCM_TAG_LENGTH = 128
    private val IV_SIZE = 12 // default for AES/GCM

    private fun encrypt(plainText: String): String {
        val cipher = try {
            Cipher.getInstance(ALGORITHM)
        } catch (_: KeyPermanentlyInvalidatedException) {
            // Key is invalid, generate a new one
            generateMasterKey()
            Cipher.getInstance(ALGORITHM)
        }
        cipher.init(Cipher.ENCRYPT_MODE, masterKey)
        val iv = cipher.iv
        val encrypted = cipher.doFinal(plainText.toByteArray(Charsets.UTF_8))
        val combined = iv + encrypted
        return Base64.encodeToString(combined, Base64.NO_WRAP)
    }

    private fun decrypt(encryptedText: String): String {
        val decoded = Base64.decode(encryptedText, Base64.NO_WRAP)
        val iv = decoded.sliceArray(0 until IV_SIZE)
        val cipherText = decoded.sliceArray(IV_SIZE until decoded.size)
        val spec = GCMParameterSpec(GCM_TAG_LENGTH, iv)

        val cipher = try {
            Cipher.getInstance(ALGORITHM)
        } catch (_: KeyPermanentlyInvalidatedException) {
            // Key is invalid, generate a new one
            generateMasterKey()
            Cipher.getInstance(ALGORITHM)
        }
        cipher.init(Cipher.DECRYPT_MODE, masterKey, spec)
        return cipher.doFinal(cipherText).toString(Charsets.UTF_8)
    }

    fun generateMasterKey(): SecretKey {
        val keyAlias = "master_key_alias"
        val keyGen = KeyGenerator.getInstance(KeyProperties.KEY_ALGORITHM_AES, "AndroidKeyStore")
        val keyGenSpec = KeyGenParameterSpec.Builder(keyAlias, KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT)
            .setBlockModes(KeyProperties.BLOCK_MODE_GCM)
            .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_NONE)
            .build()
        keyGen.init(keyGenSpec)
        return keyGen.generateKey()
    }
}
*/