package com.zell_mbc.publicartexplorer.data

import android.content.Context
import java.net.URL
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.DebugLog
import com.zell_mbc.publicartexplorer.detailview.ImageItem
import com.zell_mbc.publicartexplorer.helpers.extractWikiWord
import com.zell_mbc.publicartexplorer.helpers.resolveImageUrl
import com.zell_mbc.publicartexplorer.helpers.wikidataTagToImageUrl
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.json.JSONObject
/*
suspend fun fetchWikimediaCommonsCategoryImage(category: String, context: Context): List<ImageItem?> {
    val maxCount = 5
    val wikimediaImages = getImageUrlsFromCategory(category, maxCount)
    val provider = "Wikimedia"
    val imageItems: List<ImageItem> = ()

    // Convert to proper URL
    wikimediaImages.forEach {
        val imageUrl = getWikimediaImageUrl(it)
        imageItems.add(ImageItem(url = imageUrl, provider = provider, attribution = attribution))
        imageUrls.add(getWikimediaImageUrl(wikimediaImage ?: ""))
    }
    return if (imageUrls.isNotEmpty()) imageUrls else null
}
*/

// Takes regular url and returns encoded string with attribution and license added
suspend fun fetchImageTags(imageTag: String, context: Context): ImageItem? {
    val imageString = resolveImageUrl(imageTag)
    val provider = extractWikiWord(imageTag) ?: context.getString(R.string.unknown)

    var attribution = ""
    var imageUrl: String? = null

    val data = imageString?.split("|") ?: emptyList()

    if (data.size == 3) {
        val uploader = data[0].ifBlank { "Unknown" }
        val license = data[1].ifBlank { "Unknown license" }
        val url = data[2]

        attribution = "$provider: $uploader (License $license)"
        imageUrl = url
    } else if (!imageString.isNullOrBlank()) {
        imageUrl = imageString
    }

    return if (!imageUrl.isNullOrBlank() && imageUrl.startsWith("http"))
        ImageItem(url = imageUrl, provider = provider, attribution = attribution)
    else
        null
}

//Example "cafb0ec8-51dd-43ac-836c-8cd1f7cb8725"
suspend fun fetchPanoramaxImage(panoramaxId: String, context: Context): ImageItem? {
    return try {
        val apiUrl = "https://api.panoramax.xyz/api/search?ids=$panoramaxId"
        val json = withContext(Dispatchers.IO) { URL(apiUrl).readText() }

        val jsonObj = JSONObject(json)
        val features = jsonObj.optJSONArray("features")
        if (features == null || features.length() == 0) return null

        val feature = features.getJSONObject(0)
        val assets = feature.optJSONObject("assets")
        val properties = feature.optJSONObject("properties")
        val providers = feature.optJSONArray("providers")

        // Pick the best available image URL
        val thumbUrl = assets?.optJSONObject("thumb")?.optString("href")
        val sdUrl = assets?.optJSONObject("sd")?.optString("href")
        val hdUrl = assets?.optJSONObject("hd")?.optString("href")
        val imageUrl = thumbUrl ?: sdUrl ?: hdUrl ?: return null

        // Contributor (first producer)
        var contributorName = context.getString(R.string.unknown)
        if (providers != null) {
            for (i in 0 until providers.length()) {
                val providerObj = providers.getJSONObject(i)
                val roles = providerObj.optJSONArray("roles")
                if (roles != null) {
                    for (j in 0 until roles.length()) {
                        if (roles.getString(j) == "producer") {
                            contributorName = providerObj.optString("name")
                            break
                        }
                    }
                }
            }
        }

        // License
        val license = properties?.optString("license") ?: "Unknown license"
        val attribution = "Panoramax: $contributorName ($license)"

        ImageItem(url = imageUrl, provider = "Panoramax", attribution = attribution)

    } catch (e: Exception) {
        DebugLog.add(e.toString())
        null
    }
}

suspend fun fetchWikidataImage(id: String): ImageItem? {
    val imageUrl = wikidataTagToImageUrl(id)
    val provider = "Wikidata"
    return if (imageUrl != null && imageUrl != "NOTHING_FOUND") ImageItem(
        url = imageUrl,
        provider = provider,
        attribution = provider
    )
    else null
}


suspend fun fetchMapillaryImage(imageId: String, context: Context): ImageItem? {
    // Fetch the image URL from Mapillary API
    // Example: Rottumtaler Alphornbläser - Ochsenhausen
    return try {
        var imageUrl: String?
        val provider = "Mapillary"
        var attribution: String
        val accessToken = "MLY|9468189469959255|06c03c8accfbbe17a44f5750a146fcc4"

        //val apiUrl = "https://graph.mapillary.com/$imageId?access_token=$accessToken&fields=thumb_2048_url"
        val apiUrl = "https://graph.mapillary.com/$imageId?access_token=$accessToken&fields=thumb_2048_url,creator"
        val json = withContext(Dispatchers.IO) {
            URL(apiUrl).readText()
        }
        val jsonObj = JSONObject(json)
        val thumbUrl =jsonObj.optString("thumb_2048_url")
        imageUrl = thumbUrl

        // Extract contributor info
        val capturedBy = jsonObj.optJSONObject("creator")
        val contributorName = capturedBy?.optString("username") ?: context.getString(R.string.unknown)

        val license = "CC-BY-SA"

        attribution = "$contributorName ($provider, $license)"
        // Show the image when the real URL is available
        if (imageUrl != null) {
            ImageItem(imageUrl, provider, attribution = attribution)
        } else null

    } catch (e: Exception) {
        DebugLog.add(e.toString())
        null
    }
}