package com.zell_mbc.publicartexplorer

import android.content.Context
import android.content.Intent
import android.net.NetworkCapabilities
import androidx.compose.foundation.text.ClickableText
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLinkStyles
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.withLink
import androidx.compose.ui.text.withStyle
import com.zell_mbc.publicartexplorer.ui.theme.link
import androidx.core.net.toUri
import com.zell_mbc.publicartexplorer.data.ViewModel
import java.util.Locale
import kotlin.text.isNotEmpty
import android.net.ConnectivityManager
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.text.TextStyle

@Composable
fun createClickableUrl(url: String?, linkText: String = ""): AnnotatedString {
    val linkUrl = url ?: ""

    return buildAnnotatedString {
        withLink(LinkAnnotation.Url(url = linkUrl, styles = TextLinkStyles(style = SpanStyle(color = MaterialTheme.colorScheme.link, textDecoration = TextDecoration.Underline)))) {
            if (linkText.isNotEmpty()) append(linkText) else append(url)
        }
    }
}

@Composable
fun ClickableUrlText(
    url: String,
    urlText: String? = null,
    textBefore: String? = null,
    textAfter: String? = null,
    textStyle: TextStyle = MaterialTheme.typography.bodyMedium,
    textColor: Color = MaterialTheme.colorScheme.primary
) {
    val uriHandler = LocalUriHandler.current

    val annotatedString = buildAnnotatedString {
        textBefore?.let {
            withStyle(style = SpanStyle(color = textColor, fontSize = textStyle.fontSize)) {
                append(it)
            }
        }

        pushStringAnnotation(tag = "URL", annotation = url)
        withStyle(
            style = SpanStyle(color = MaterialTheme.colorScheme.link, textDecoration = TextDecoration.Underline, fontSize = textStyle.fontSize)
        ) {
            append(urlText ?: url)
        }
        pop()

        textAfter?.let {
            withStyle(style = SpanStyle(color = textColor, fontSize = textStyle.fontSize)) {
                append(it)
            }
        }
    }

    ClickableText(
        text = annotatedString,
        style = textStyle,
        onClick = { offset ->
            annotatedString.getStringAnnotations(tag = "URL", start = offset, end = offset)
                .firstOrNull()?.let { annotation ->
                    uriHandler.openUri(annotation.item)
                }
        }
    )
}

@Composable
fun ClickableEmail(email: String) {
    val context = LocalContext.current
    val annotatedString = buildAnnotatedString {
        // Add the email as an annotated string
        pushStringAnnotation(tag = "EMAIL", annotation = email)
        withStyle(style = SpanStyle(color = MaterialTheme.colorScheme.link)) { // blue color
            append(email)
        }
        pop()
    }

    ClickableText(
        text = annotatedString,
        onClick = { offset: Int ->
            annotatedString.getStringAnnotations(tag = "EMAIL", start = offset, end = offset)
                .firstOrNull()?.let { annotation ->
                    val intent = Intent(Intent.ACTION_SENDTO).apply {
                        data = "mailto:${annotation.item}".toUri()
                    }
                    context.startActivity(intent)
                }
        }    )
}

fun capitalize(input: String) = input.replaceFirstChar { it.titlecase(Locale.getDefault()) }


fun downloadArtworkList(context: Context, viewModel: ViewModel) {
    if (viewModel.artworks.isEmpty()) return

    val sb = StringBuilder()
    val separator = ","
    var line = "longitude,latitude,type" + System.lineSeparator()
    sb.append(line)

    for (item in viewModel.artworks) {
        val type = item.tags["artwork_type"] ?: ""
        val name =  item.tags["name"] ?: "No name"
        line = item.lon.toString() + separator + item.lat.toString() + separator + type + separator + name + System.lineSeparator()
        sb.append(line)
    }

    val sendIntent = Intent().apply {
        action = Intent.ACTION_SEND
        putExtra(Intent.EXTRA_TEXT, sb.toString())
        type = "text/plain"
    }

    val shareIntent = Intent.createChooser(sendIntent, "Share via")
    context.startActivity(shareIntent)
}

fun currentLocale(context: Context): String {
    val currentLocale = context.resources.configuration.locales.get(0)
    return currentLocale.language // e.g., "en", "fr", "de"
}

// Iterate over language tags and return either the first one or the entry for the current language
fun getLocalValue(key: String, tags: MutableMap<String, String>, context: Context ): String {
    val currentLanguage = currentLocale(context)
    var first = ""
    var local = ""
    for (tag in tags.entries.reversed()) {
        val t = tag.key
        if (t.contains(key)) {
            if (t.contains(":$currentLanguage")) local = tag.value
            else first  = tag.value
        }
    }
    return local.ifEmpty { first }
}


// I can't show images from Google Photos (or similar), only option is as website
fun isPhotoAlbum(url: String): String? {
    return when {
        "photos.app.goo.gl" in url -> "Google Photos"
        "flickr.com" in url && !url.contains("/static/") -> "Flickr"
        "instagram.com" in url -> "Instagram"
        else -> null
    }
}

fun isNetworkAvailable(context: Context): Boolean {
    val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
    val network = connectivityManager.activeNetwork ?: return false
    val capabilities = connectivityManager.getNetworkCapabilities(network) ?: return false
    return capabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)
}

fun nanoToTime(elapsedNanos: Long): String {
    val elapsedMillis = elapsedNanos / 1_000_000
    val humanTime = when {
        elapsedMillis < 1000 -> "$elapsedMillis ms"
        elapsedMillis < 60_000 -> "${"%.2f".format(elapsedMillis / 1000.0)} s"
        else -> {
            val minutes = elapsedMillis / 60000
            val seconds = (elapsedMillis % 60000) / 1000
            "${minutes}m ${seconds}s"
        }
    }
    return humanTime
}
