package com.zell_mbc.publicartexplorer

import android.content.Context
import android.location.Geocoder
import android.os.Build
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.size
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.hapticfeedback.HapticFeedbackType
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalHapticFeedback
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.datastore.core.IOException
import com.zell_mbc.publicartexplorer.data.ViewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

@Composable
fun AddressInput(viewModel: ViewModel, onSearchComplete: (lat: Double, lon: Double) -> Unit) {
    val context = LocalContext.current
    val preferences = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
    val enableHapticFeedback = preferences.getBoolean(ENABLE_HAPTIC_FEEDBACK_KEY, true)
    val hapticFeedback = LocalHapticFeedback.current

    var query by remember { mutableStateOf("") }
    var searchInProgress by remember { mutableStateOf(false) }
    val coroutineScope = rememberCoroutineScope()
    var latitude = 0.0
    var longitude = 0.0

    AlertDialog(
        onDismissRequest = {
            // Supply some fallback values or only call if you have valid coordinates
            val lat = latitude
            val lon = longitude
            onSearchComplete(lat, lon)
        },
        title = { Text(stringResource(R.string.searchAddress)) },
        text = {
            Column {
                TextField(value = query, onValueChange = { query = it }, label = { Text(stringResource(R.string.address)) }, enabled = !searchInProgress)
            }
        },
        confirmButton = { if (searchInProgress)  // Show spinner instead of button
                Box(modifier = Modifier.fillMaxWidth(), contentAlignment = Alignment.Center) {
                    CircularProgressIndicator(modifier = Modifier.size(48.dp), color = MaterialTheme.colorScheme.primary)
                }
            else {
                Button(
                    onClick = {
                        if (query.isEmpty()) {
                            viewModel.showMessage(context.getString(R.string.noAddressFound))
                            //Toast.makeText(context, context.getString(R.string.noAddressFound), Toast.LENGTH_LONG).show()
                            return@Button
                        }
                        searchInProgress = true
                        coroutineScope.launch {
                            try {
                                val geocoder = Geocoder(context)
                                // If suspend version available, call directly
                                // Otherwise, wrap blocking call in IO dispatcher
                                val addresses = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                                    geocoder.getFromLocationName(query, 1) // suspend, safe
                                } else {
                                    withContext(Dispatchers.IO) {
                                        geocoder.getFromLocationName(query, 1) // blocking, move off Main
                                    }
                                }
                                //val addresses = geocoder.getFromLocationNameCompat(query, 1)

                                if (!addresses.isNullOrEmpty()) {
                                    val address = addresses[0]
                                    onSearchComplete(address.latitude, address.longitude)
                                } else {
                                    viewModel.showMessage(context.getString(R.string.noAddressFound))
                                    //Toast.makeText(context, context.getString(R.string.noAddressFound), Toast.LENGTH_LONG).show()
                                }
                            } catch (e: IOException) {
                                viewModel.showMessage( context.getString(R.string.error) + " " + e.message)
                                //Toast.makeText(context, context.getString(R.string.error) + " " + e.message, Toast.LENGTH_LONG).show()
                            } finally {
                                searchInProgress = false
                            }
                        }
                    }
                ) { Text(stringResource(R.string.search)) }
            }
        },
        dismissButton = {
            OutlinedButton(
                onClick = {
                    if (enableHapticFeedback) hapticFeedback.performHapticFeedback(HapticFeedbackType.Confirm)
                    onSearchComplete(0.0, 0.0) },
                enabled = !searchInProgress
            ) {
                Text(stringResource(R.string.cancel))
            }
        }
    )
}