package com.zell_mbc.publicartexplorer.billing

import android.app.Activity
import android.content.Context
import android.content.SharedPreferences
import android.os.Handler
import android.os.Looper
import android.preference.PreferenceManager
import android.util.Log
import android.widget.Toast
import com.android.billingclient.api.AcknowledgePurchaseParams
import com.android.billingclient.api.BillingClient
import com.android.billingclient.api.BillingClientStateListener
import com.android.billingclient.api.BillingFlowParams
import com.android.billingclient.api.BillingResult
import com.android.billingclient.api.PendingPurchasesParams
import com.android.billingclient.api.QueryProductDetailsParams
import com.android.billingclient.api.QueryPurchasesParams
import androidx.core.content.edit
import com.android.billingclient.api.Purchase
import com.zell_mbc.publicartexplorer.DebugLog
import com.zell_mbc.publicartexplorer.EXPERT_MODE_KEY
import com.zell_mbc.publicartexplorer.SUBSCRIPTION_ACTIVE_KEY
import java.time.Instant
import java.time.ZoneId
import java.time.ZonedDateTime
import java.time.format.DateTimeFormatter
import java.util.Calendar

class BillingManagerImpl() : BillingManager {
    private var billingClient: BillingClient? = null
    private var isBillingReady = false
    private var pendingCheck: (() -> Unit)? = null
    private lateinit var preferences: SharedPreferences
    private var purchasingProcessActive = false

    // Callback variables
    override var showThankYouDialog: ((Boolean) -> Unit)? = null
    override var setSubscriptionStatus: ((Boolean) -> Unit)? = null
    override var showBillingUnavailableDialog: ((Boolean) -> Unit)? = null
    override var setExpiryTimestamp: ((Long) -> Unit)? = null

    private fun getBillingClient(context: Context): BillingClient {
        if (billingClient == null) {
            billingClient = BillingClient.newBuilder(context)
                .setListener { billingResult, purchases ->
                    preferences = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
                    when (billingResult.responseCode) {
                    BillingClient.BillingResponseCode.OK  -> {
                      if (purchases != null)
                        for (purchase in purchases) {
                            if (purchase.products.contains(SUBSCRIPTION_ACTIVE_KEY)) {
                                // Unlock premium features
                                preferences.edit { putBoolean(SUBSCRIPTION_ACTIVE_KEY, true) }

                                // Acknowledge purchase if not yet acknowledged
                                if (purchase.purchaseState == Purchase.PurchaseState.PURCHASED && !purchase.isAcknowledged) {
                                    val acknowledgeParams = AcknowledgePurchaseParams.newBuilder().setPurchaseToken(purchase.purchaseToken).build()
                                    billingClient?.acknowledgePurchase(acknowledgeParams) { ackResult ->
                                        if (ackResult.responseCode == BillingClient.BillingResponseCode.OK) {
                                            preferences.edit { putBoolean(SUBSCRIPTION_ACTIVE_KEY, true) }
                                            Log.d("Billing", "Purchase acknowledged")
                                            DebugLog.add("Purchase acknowledged")
                                            showThankYouDialog?.invoke(true)
                                            setSubscriptionStatus?.invoke(true)
                                        } else {
                                            val message = "Error acknowledging purchase: ${ackResult.debugMessage}"
                                            Log.w("Billing", message)
                                            DebugLog.add(message)
                                            //Handler(Looper.getMainLooper()).post { Toast.makeText(context, message, Toast.LENGTH_LONG).show() }
                                        }
                                    }
                                }
                            }
                        }
                    }
                    BillingClient.BillingResponseCode.ITEM_ALREADY_OWNED -> {
                        Log.w("Billing", "Item already owned")
                        preferences.edit { putBoolean(SUBSCRIPTION_ACTIVE_KEY, true) }
                        setSubscriptionStatus?.invoke(true)
                        }
                    else ->{
                        Log.w("Billing", "Purchase failed: ${billingResult}")
                        preferences.edit { putBoolean(SUBSCRIPTION_ACTIVE_KEY, false) }
                        setSubscriptionStatus?.invoke(false)
                        Handler(Looper.getMainLooper()).post { Toast.makeText(context, "Unknown error: ${billingResult}", Toast.LENGTH_LONG).show() }
                        }
                    }
                }
                .enablePendingPurchases(PendingPurchasesParams.newBuilder().enableOneTimeProducts().build())
                .enableAutoServiceReconnection()
                .build()

            billingClient?.startConnection(object : BillingClientStateListener {
                override fun onBillingSetupFinished(billingResult: BillingResult) {
                    if (billingResult.responseCode == BillingClient.BillingResponseCode.OK) {
                        Log.d("Billing", "Billing setup finished")
                        isBillingReady = true
                        pendingCheck?.invoke()
                        pendingCheck = null
                    } else {
                        // Billing is not available: show appropriate message to user
                        Log.w("Billing", "Billing unavailable: ${billingResult.responseCode} (${billingResult.debugMessage})")
                        if (purchasingProcessActive) {
                            showBillingUnavailableDialog?.invoke(true)
                            purchasingProcessActive = false
                        } // Ignore otherwise
                    }
                }

                override fun onBillingServiceDisconnected() {
                    Log.d("Billing", "Billing service disconnected")
                    isBillingReady = false
                }
            })
        }

        return billingClient!!
    }


    override fun isSubscriptionActive(context: Context) {
        var subscriptionActive = false
        val billingClient = getBillingClient(context)

        if (!isBillingReady) {
            // Defer the check until ready
            pendingCheck = { isSubscriptionActive(context) }
            Log.d("Billing", "Not ready")
            return
        }
        billingClient.queryPurchasesAsync(
            QueryPurchasesParams.newBuilder()
                .setProductType(BillingClient.ProductType.SUBS)
                .build()
        ) { billingResult, purchasesList ->
            subscriptionActive = purchasesList.any { it.products.contains(SUBSCRIPTION_ACTIVE_KEY) }
            val purchase = purchasesList.firstOrNull { it.products.contains(SUBSCRIPTION_ACTIVE_KEY) }
            if (purchase != null) {
                val purchaseTimeMillis = purchase.purchaseTime
                val calendar = Calendar.getInstance()
                calendar.timeInMillis = purchaseTimeMillis
                calendar.add(Calendar.YEAR, 1) // Add one year for expiry date
                setExpiryTimestamp?.invoke(calendar.timeInMillis)
            }
            preferences = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
            preferences.edit { putBoolean(SUBSCRIPTION_ACTIVE_KEY, subscriptionActive) } // Can be true or false!
            //preferences.edit { putBoolean(EXPERT_MODE_KEY, subscriptionActive) } // Make sure to disable ExpertMode in case sub expired
            setSubscriptionStatus?.invoke(subscriptionActive)
            Log.d("Billing", "$billingResult, ExpertMode=$subscriptionActive")
        }
        return
    }

    // Process purchase
    override fun launchPurchaseFlow(activity: Activity) {
        val queryProductDetailsParams = QueryProductDetailsParams.newBuilder()
            .setProductList(
                listOf(
                    QueryProductDetailsParams.Product.newBuilder()
                        .setProductId(SUBSCRIPTION_ACTIVE_KEY) // your product ID here
                        .setProductType(BillingClient.ProductType.SUBS)
                        .build()
                )
            )
            .build()

        billingClient?.queryProductDetailsAsync(queryProductDetailsParams) { billingResult, queryProductDetailsResult ->
            if (billingResult.responseCode == BillingClient.BillingResponseCode.OK) {
                val productDetailsList = queryProductDetailsResult.productDetailsList
                if (productDetailsList.isNotEmpty()) {
                    val productDetails = productDetailsList[0]
                    // Get the offer token from the first subscription offer
                    val offerToken = productDetails.subscriptionOfferDetails
                        ?.firstOrNull()
                        ?.offerToken ?: ""

                    val productDetailsParams = BillingFlowParams.ProductDetailsParams.newBuilder()
                        .setProductDetails(productDetails)
                        .setOfferToken(offerToken)  // This is required for subscriptions
                        .build()

                    // Proceed with purchase flow
                    val billingFlowParams = BillingFlowParams.newBuilder()
                        .setProductDetailsParamsList(listOf(productDetailsParams))
                        .build()

                    // This will trigger the billingClient.setListener defined above
                    billingClient?.launchBillingFlow(activity, billingFlowParams)
                }
                // Optionally handle unfetched products:
                val unfetchedList = queryProductDetailsResult.unfetchedProductList
                if (unfetchedList.isNotEmpty()) {
                    Log.w("Billing", "Some products could not be fetched: $unfetchedList")
                    // Optionally notify the user or fallback gracefully
                    //showBillingErrorToUser("Unable to fetch all products. Try again later.")
                }
            }
        }
    }

    // Check if unlimitedRecords product is available, if so, return product details
    override fun queryUpgradeProduct(context: Context, onResult: (UpgradeProductInfo?) -> Unit) {
        val params = QueryProductDetailsParams.newBuilder()
            .setProductList(
                listOf(
                    QueryProductDetailsParams.Product.newBuilder()
                        .setProductId(SUBSCRIPTION_ACTIVE_KEY)
                        .setProductType(BillingClient.ProductType.SUBS)
                        .build()
                )
            )
            .build()
        purchasingProcessActive = true
        billingClient?.queryProductDetailsAsync(params) { billingResult, result ->
            val details = result.productDetailsList.firstOrNull()
            if (details != null) {
                val price = details.subscriptionOfferDetails
                    ?.firstOrNull()
                    ?.pricingPhases
                    ?.pricingPhaseList
                    ?.firstOrNull()
                    ?.formattedPrice ?: "N/A"  // Fallback price string
                onResult(
                    UpgradeProductInfo(
                        title = details.title,
                        description = details.description,
                        price = price
                    )
                )
            } else {
                Handler(Looper.getMainLooper()).post { Toast.makeText(context, billingResult.debugMessage, Toast.LENGTH_LONG).show() }
                onResult(null)
            }
        }
    }
}
