package com.zell_mbc.publicartexplorer.billing

import androidx.activity.compose.LocalActivity
import androidx.compose.foundation.Image
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.window.Dialog
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Star
import androidx.compose.material.icons.filled.Warning
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLinkStyles
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.withLink
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.zell_mbc.publicartexplorer.ProjectDetails
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.currentLocale
import com.zell_mbc.publicartexplorer.data.ViewModel
import com.zell_mbc.publicartexplorer.screens.SettingButton
import com.zell_mbc.publicartexplorer.ui.theme.link


@Composable
fun HandleSubscription(viewModel: ViewModel) {
    val context = LocalContext.current
    val activity = LocalActivity.current

    var productDetails by remember { mutableStateOf<UpgradeProductInfo?>(null) }

    Column {
        Spacer(Modifier.height(16.dp))
        SettingButton(
            title = stringResource(R.string.upgrade),
            summary = stringResource(R.string.upgradeSummary),
            buttonText = stringResource(R.string.proceed),
            onClick = {
                viewModel.billingManager.queryUpgradeProduct(context) { details -> productDetails = details }
                viewModel.showUpgradeDialogState = true
            }
        )
        Text("")

        if (viewModel.showUpgradeDialogState && activity != null){
            ExpertModeUpgradeDialog(
                show = viewModel.showUpgradeDialogState,
                productDetails = productDetails,
                onPurchaseClick = { pd ->
                    viewModel.billingManager.launchPurchaseFlow(activity)
                    viewModel.showUpgradeDialogState = false
                },
                onDismiss = { viewModel.showUpgradeDialogState = false }
            )
        }
    }
}

@Composable
fun ExpertModeUpgradeDialog(
    show: Boolean,
    productDetails: UpgradeProductInfo?,
    onPurchaseClick: (UpgradeProductInfo) -> Unit,
    onDismiss: () -> Unit
) {
    if (show && productDetails != null) {
        val context = LocalContext.current

        Dialog(onDismissRequest = onDismiss) {
            Surface(
                shape = MaterialTheme.shapes.medium,
                tonalElevation = 8.dp
            ) {
                Column(modifier = Modifier.padding(24.dp), horizontalAlignment = Alignment.CenterHorizontally) {
                    Image(painter = painterResource(id = if (isSystemInDarkTheme()) R.drawable.featuregraphic_dark else R.drawable.featuregraphic_light),
                        contentDescription = stringResource(id = R.string.appName),
                        modifier = Modifier.size(200.dp))

                    Text(text = stringResource(R.string.upgrade), style = MaterialTheme.typography.titleLarge)
                    Spacer(Modifier.height(8.dp))
                    Text(text = stringResource(R.string.licenseType), style = MaterialTheme.typography.bodyLarge)
                    Spacer(Modifier.height(8.dp))
                    Text(text = stringResource(R.string.price) + ": " + productDetails.price, style = MaterialTheme.typography.bodyMedium)
                    Spacer(Modifier.height(16.dp))
                    // Why subscribe
                    var url = ProjectDetails.EXPERT_MODE_URL
                    val currentLanguage = currentLocale(context)
                    url += when (currentLanguage) {
                        "de" -> "/de/expertenmodus"
                        else -> "/en/expertmode"
                    }
                    Text(
                        buildAnnotatedString {
                            append("")
                            withLink(LinkAnnotation.Url(url = url, TextLinkStyles(style = SpanStyle(color = MaterialTheme.colorScheme.link)))
                            ) { append(stringResource(R.string.whySubscribe)) }
                        },
                        style = MaterialTheme.typography.bodyMedium,
                        textDecoration = TextDecoration.Underline
                    )
                    Spacer(Modifier.height(16.dp))
                    HorizontalDivider(thickness = 1.dp)
                    Spacer(Modifier.height(16.dp))

                    Button(onClick = { onPurchaseClick(productDetails) }, modifier = Modifier.fillMaxWidth()) {
                        Text(stringResource(R.string.unlockExpertMode))
                    }
                    Spacer(Modifier.height(8.dp))
                    OutlinedButton(onClick = onDismiss, modifier = Modifier.fillMaxWidth()
                    ) {
                        Text(stringResource(R.string.cancel))
                    }
                }
            }
        }
    }
}

@Composable
fun ThankYouForPurchaseDialog(
    onDismiss: () -> Unit
) {
    AlertDialog(
        onDismissRequest = onDismiss,
        title = {
            Text(text = stringResource(R.string.thankYou), style = MaterialTheme.typography.headlineSmall, color = MaterialTheme.colorScheme.primary)
        },
        text = {
            Column(horizontalAlignment = Alignment.CenterHorizontally) {
                Icon(
                    imageVector = Icons.Default.Star,
                    contentDescription = null,
                    tint = MaterialTheme.colorScheme.secondary,
                    modifier = Modifier.size(48.dp)
                )
                Spacer(modifier = Modifier.height(12.dp))
                Text(stringResource(R.string.thankYouDescription), textAlign = TextAlign.Center, style = MaterialTheme.typography.bodyLarge)
            }
        },
        confirmButton = {
            Button(onClick = onDismiss) {
                Text(stringResource(R.string.ok))
            }
        },
        shape = RoundedCornerShape(20.dp),
        tonalElevation = 8.dp
    )
}


@Composable
fun BillingUnavailableDialog(
    onDismiss: () -> Unit
) {
    AlertDialog(
        onDismissRequest = onDismiss,
        icon = {
            Icon(
                imageVector = Icons.Default.Warning,
                contentDescription = stringResource(R.string.warning),
                tint = Color(0xFFFFC107), // Amber
                modifier = Modifier.size(48.dp)
            )
        },
        title = {
            Text(stringResource(R.string.googlePlayRequired), fontWeight = FontWeight.Bold, fontSize = 20.sp)
        },
        text = {
            Text(stringResource(R.string.googlePlayRequiredDescription), fontSize = 16.sp)
        },
        confirmButton = {
            Button(onClick = onDismiss) {
                Text(stringResource(R.string.ok))
            }
        }
    )
}
