package com.zell_mbc.publicartexplorer.wiki

import android.content.Context
import android.content.Intent
import android.location.Location
import android.net.Uri
import android.provider.OpenableColumns
import android.util.Log
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.runtime.Composable
import com.zell_mbc.publicartexplorer.BuildConfig
import com.zell_mbc.publicartexplorer.appVersion
import com.zell_mbc.publicartexplorer.sharedOkHttpClient
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withContext
import okhttp3.FormBody
import okhttp3.HttpUrl
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.MultipartBody
import okhttp3.Request
import okhttp3.RequestBody
import okio.BufferedSink
import okio.source
import org.json.JSONObject
import java.io.IOException
import java.net.SocketTimeoutException
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

suspend fun uploadImageToCommons(
    context: Context,
    accessToken: String,
    csrfToken: String,
    uri: Uri?,
    categories: List<String> = emptyList(),
    description: String,
    caption: String): String? = withContext(Dispatchers.IO) {

    if (uri == null) return@withContext null

    val filenameOnCommons = "PublicArtExplorer_${System.currentTimeMillis()}.jpg"
    val mimeType = context.contentResolver.getType(uri) ?: "image/jpeg"

    val fileSize = try {
        context.contentResolver.query(uri, arrayOf(OpenableColumns.SIZE), null, null, null)?.use { cursor ->
            if (cursor.moveToFirst()) cursor.getLong(0) else -1L
        } ?: -1L
    } catch (e: Exception) { -1L }

    val comment = "Uploaded with PublicArtExplorer for Android " + if (BuildConfig.FLAVOUR == "foss") "(FOSS)" else "(GooglePlay)"
    val user = if (WikiUserObject.name.isEmpty()) "PublicArtExplorer" else WikiUserObject.name
    val dateString = getImageCreationDateString(context, uri)

    val localCategories = categories + "Uploaded with PublicArtExplorer App"
    val categoryText = localCategories.joinToString("\n") { "[[Category:${it.trim()}]]" }

    val descriptionText = """
=={{int:filedesc}}==
{{Information
|description={{en|1=$description}}
|date=$dateString
|source={{Self-photographed}}
|author=[[User:$user]]
|permission={{Cc-by-sa-4.0}}
}}
$categoryText
""".trimIndent()

    val multipart = MultipartBody.Builder().setType(MultipartBody.FORM)
        .addFormDataPart("action", "upload")
        .addFormDataPart("filename", filenameOnCommons)
        .addFormDataPart("comment", comment)
        .addFormDataPart("token", csrfToken)
        .addFormDataPart("ignorewarnings", "1")
        .addFormDataPart("format", "json")
        .addFormDataPart("text", descriptionText)
        .addFormDataPart(
            "file",
            filenameOnCommons,
            object : RequestBody() {
                override fun contentType() = mimeType.toMediaTypeOrNull()
                override fun contentLength() = if (fileSize > 0) fileSize else -1L
                override fun writeTo(sink: BufferedSink) {
                    context.contentResolver.openInputStream(uri)?.use { input ->
                        sink.writeAll(input.source())
                    } ?: throw IOException("Unable to open input for $uri")
                }
            }
        ).build()

    val request = Request.Builder()
        .url("https://commons.wikimedia.org/w/api.php")
        .header("Authorization", "Bearer $accessToken")
        .header("User-Agent", "PublicArtExplorer/$appVersion (publicartexplorer@zell-mbc.com)")
        .post(multipart)
        .build()

    // Execute the upload in a cancellation-aware way
    val fileOnCommons = suspendCancellableCoroutine<String?> { cont ->
        val call = sharedOkHttpClient.newCall(request)

        cont.invokeOnCancellation { try { call.cancel() } catch (_: Exception) {} }

        try {
            call.execute().use { resp ->
                val bodyStr = try { resp.body.string() } catch (e: Exception) { null }

                if (!resp.isSuccessful) {
                    Log.e("Upload", "HTTP ${resp.code}, body=$bodyStr")
                    cont.resume(null)
                    return@use
                }

                if (bodyStr.isNullOrEmpty()) {
                    Log.e("Upload", "Empty response body")
                    cont.resume(null)
                    return@use
                }

                try {
                    val json = JSONObject(bodyStr)
                    val uploadObj = json.optJSONObject("upload")
                    if (uploadObj?.optString("result") == "Success") {
                        val fileOnCommons = uploadObj.optString("filename")
                        cont.resume(fileOnCommons)
                    } else {
                        Log.e("Upload", "Upload failed, body=$bodyStr")
                        cont.resume(null)
                    }
                } catch (je: Exception) {
                    Log.e("Upload", "JSON parse error", je)
                    cont.resume(null)
                }
            }
        } catch (ioe: SocketTimeoutException) {
            Log.e("Upload", "Upload timeout", ioe)
            if (!cont.isCompleted) cont.resume(null)
        } catch (ioe: IOException) {
            // network error or cancellation
            Log.e("Upload", "IO error during upload", ioe)
            if (!cont.isCompleted) cont.resume(null)
        } catch (t: Throwable) {
            Log.e("Upload", "Unexpected error", t)
            if (!cont.isCompleted) cont.resumeWithException(t)
        }
    }

    fileOnCommons?.let { filename ->
        CoroutineScope(Dispatchers.IO).launch {
            setCommonsCaption(accessToken = accessToken, csrfToken = csrfToken, filename = filename, caption = caption)
        }
    }

    return@withContext fileOnCommons
}

suspend fun setCommonsCaptionAndLicense(
    accessToken: String,
    csrfToken: String,
    filename: String,
    caption: String,
    licenseQid: String = "Q18199165" // CC BY-SA 4.0 International
) = withContext(Dispatchers.IO) {
    try {
        // 1️⃣ Get MediaInfo ID for the uploaded file
        val queryUrl = HttpUrl.Builder()
            .scheme("https")
            .host("commons.wikimedia.org")
            .addPathSegment("w")
            .addPathSegment("api.php")
            .addQueryParameter("action", "query")
            .addQueryParameter("titles", "File:$filename")
            .addQueryParameter("prop", "pageprops")
            .addQueryParameter("format", "json")
            .build()

        val queryRequest = Request.Builder()
            .url(queryUrl)
            .header("Authorization", "Bearer $accessToken")
            .build()

        val mediaInfoId = sharedOkHttpClient.newCall(queryRequest).execute().use { resp ->
            if (!resp.isSuccessful) {
                Log.e("Commons", "Failed to get MediaInfo ID: ${resp.code}")
                return@withContext
            }
            val body = resp.body.string()
            val pages = JSONObject(body).getJSONObject("query").getJSONObject("pages")
            val firstPage = pages.keys().asSequence().firstOrNull()?.let { pages.getJSONObject(it) }
            "M${firstPage?.optLong("pageid")}"
        }

        if (mediaInfoId.length <= 1) { // The "M" is always there
            Log.e("Commons", "No MediaInfo ID found for $filename: $mediaInfoId")
            return@withContext
        }

        // 2️⃣ Set the caption
        val captionBody = FormBody.Builder()
            .add("action", "wbsetlabel")
            .add("id", mediaInfoId)
            .add("language", "en")
            .add("value", caption)
            .add("token", csrfToken)
            .add("format", "json")
            .build()

        val captionRequest = Request.Builder()
            .url("https://commons.wikimedia.org/w/api.php")
            .header("Authorization", "Bearer $accessToken")
            .post(captionBody)
            .build()

        sharedOkHttpClient.newCall(captionRequest).execute().use { resp ->
            if (!resp.isSuccessful) {
                Log.e("Commons", "Failed to set caption: ${resp.code}")
            } else {
                Log.d("Commons", "Caption successfully set for $filename")
            }
        }

        // 3️⃣ Set the structured license (P6216)
        val claimsJson = JSONObject().apply {
            put("mainsnak", JSONObject().apply {
                put("snaktype", "value")
                put("property", "P6216")
                put("datavalue", JSONObject().apply {
                    put("value", JSONObject().apply { put("id", licenseQid) })
                    put("type", "wikibase-entityid")
                })
            })
            put("type", "statement")
            put("rank", "normal")
        }

        val dataJson = JSONObject().apply {
            put("claims", listOf(claimsJson))
        }

        val licenseBody = FormBody.Builder()
            .add("action", "wbeditentity")
            .add("id", mediaInfoId)
            .add("data", dataJson.toString())
            .add("token", csrfToken)
            .add("format", "json")
            .build()

        val licenseRequest = Request.Builder()
            .url("https://commons.wikimedia.org/w/api.php")
            .header("Authorization", "Bearer $accessToken")
            .post(licenseBody)
            .build()

        sharedOkHttpClient.newCall(licenseRequest).execute().use { resp ->
            if (!resp.isSuccessful) {
                Log.e("Commons", "Failed to set license: ${resp.code}")
            } else {
                Log.d("Commons", "License successfully set for $filename")
            }
        }

    } catch (e: Exception) {
        Log.e("Commons", "Caption/License error", e)
    }
}


suspend fun setCommonsCaption(
    accessToken: String,
    csrfToken: String,
    filename: String,
    caption: String
) = withContext(Dispatchers.IO) {
    try {
        // 1️⃣ Get MediaInfo ID for the uploaded file
        val queryUrl = HttpUrl.Builder()
            .scheme("https")
            .host("commons.wikimedia.org")
            .addPathSegment("w")
            .addPathSegment("api.php")
            .addQueryParameter("action", "query")
            .addQueryParameter("titles", "File:$filename")
            .addQueryParameter("prop", "pageprops")
            .addQueryParameter("format", "json")
            .build()

        val queryRequest = Request.Builder()
            .url(queryUrl)
            .header("Authorization", "Bearer $accessToken")
            .build()

        val mediaInfoId = sharedOkHttpClient.newCall(queryRequest).execute().use { resp ->
            if (!resp.isSuccessful) {
                Log.e("Commons", "Failed to get MediaInfo ID: ${resp.code}")
                return@withContext
            }
            val body = resp.body.string()
            val pages = JSONObject(body).getJSONObject("query").getJSONObject("pages")
            val firstPage = pages.keys().asSequence().firstOrNull()?.let { pages.getJSONObject(it) }
            "M${firstPage?.optLong("pageid")}"
        }

        if (mediaInfoId.length <= 1) { // The "M" is always there
            Log.e("Commons", "No MediaInfo ID found for $filename")
            return@withContext
        }

        // 2️⃣ Now set the caption
        val captionBody = FormBody.Builder()
            .add("action", "wbsetlabel")
            .add("id", mediaInfoId)
            .add("language", "en")
            .add("value", caption)
            .add("token", csrfToken)
            .add("format", "json")
            .build()

        val captionRequest = Request.Builder()
            .url("https://commons.wikimedia.org/w/api.php")
            .header("Authorization", "Bearer $accessToken")
            .post(captionBody)
            .build()

        sharedOkHttpClient.newCall(captionRequest).execute().use { resp ->
            if (!resp.isSuccessful) {
                Log.e("Commons", "Failed to set caption: ${resp.code}")
            } else {
                Log.d("Commons", "Caption successfully set for $filename")
            }
        }

    } catch (e: Exception) {
        Log.e("Commons", "Caption error", e)
    }
}

fun getCsrfToken(accessToken: String): String? {
    val client = sharedOkHttpClient

    val request = Request.Builder()
        .url("https://commons.wikimedia.org/w/api.php?action=query&meta=tokens&type=csrf&format=json")
        .header("Authorization", "Bearer $accessToken")
        .build()

    client.newCall(request).execute().use { response ->
        if (!response.isSuccessful) return null

        val json = JSONObject(response.body.string())
        return json
            .getJSONObject("query")
            .getJSONObject("tokens")
            .getString("csrftoken")
    }
}

// Using the document based file picker to work around a Google bug which strips GPS data from images :-(
@Composable
fun rememberGpsSafeImagePicker(
    context: Context,
    onImagePicked: (Uri?, Location?) -> Unit
): ActivityResultLauncher<Intent> {

    return rememberLauncherForActivityResult(
        contract = ActivityResultContracts.StartActivityForResult(),
        onResult = { result ->
            val uri = result.data?.data
            if (uri != null) {
                // Persist permission for future access
                context.contentResolver.takePersistableUriPermission(
                    uri,
                    Intent.FLAG_GRANT_READ_URI_PERMISSION
                )
                dumpExif(context, uri)
                // Extract GPS
                val location = extractGpsFromImageUri(context, uri).let { imgLoc ->
                    Location("exif").apply {
                        latitude = imgLoc.latitude ?: 0.0
                        longitude = imgLoc.longitude ?: 0.0
                    }
                }

                onImagePicked(uri, location)
            } else {
                onImagePicked(null, null)
            }
        }
    )
}
