package com.zell_mbc.publicartexplorer.screens

import android.widget.Toast
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.Button
import androidx.compose.material3.Checkbox
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.MenuAnchorType
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Slider
import androidx.compose.material3.Switch
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Alignment.Companion.CenterVertically
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.text.input.VisualTransformation
import androidx.compose.ui.unit.dp
import com.zell_mbc.publicartexplorer.OSM_ACCESS_TOKEN_KEY
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.oauth2.startOsmLogin
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers

@Composable
fun SettingSlider(
    modifier: Modifier = Modifier,
    label: String,
    value: Float,
    valueRange: ClosedFloatingPointRange<Float>,
    unit: String = "km", // default unit
    onValueChange: (Float) -> Unit,
) {
    // Helper to format the value with unit and scale display
    val displayText = when (unit) {
        "m" -> "${value.toInt()} m"
        "km" -> { if (value >= 1000f) "${(value / 1000).format(1)} km" else "${value.toInt()} m" }
        "mi" -> {
            val miles = value * 0.000621371f
            if (miles >= 1f) "${miles.format(1)} mi" else "${(miles * 5280).toInt()} ft"
        }
        else -> "${value.toInt()} $unit"
    }

    Column(modifier = modifier.padding(vertical = 8.dp)) {
        Text(text = "$label: $displayText", style = MaterialTheme.typography.titleMedium)
        Slider(
            value = value,
            onValueChange = onValueChange,
            valueRange = valueRange,
            steps = ((valueRange.endInclusive - valueRange.start) / 100).toInt() - 1,
            modifier = Modifier.fillMaxWidth()
        )
    }
}

// Extension to format floats nicely
fun Float.format(digits: Int) = "%.${digits}f".format(this)


@Composable
fun SettingSwitch(
    title: String,
    summary: String,
    checked: Boolean,
    onCheckedChange: (Boolean) -> Unit,
    modifier: Modifier = Modifier
) {
    Column(modifier = modifier.padding(vertical = 8.dp)) {
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = CenterVertically
        ) {
            Column(modifier = Modifier.weight(1f)) {
                Text(title, style = MaterialTheme.typography.titleMedium)
                Text(summary, style = MaterialTheme.typography.bodySmall, color = MaterialTheme.colorScheme.onSurfaceVariant)
            }
            Switch(checked = checked, onCheckedChange = onCheckedChange)
        }
    }
}

@Composable
fun SettingButton(
    title: String,
    summary: String,
    buttonText: String,
    onClick: () -> Unit,
    modifier: Modifier = Modifier
) {
    Column(modifier = modifier.padding(vertical = 8.dp)) {
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = CenterVertically
        ) {
            Column(modifier = Modifier.weight(1f)) {
                Text(title, style = MaterialTheme.typography.titleMedium)
                Text(summary, style = MaterialTheme.typography.bodySmall, color = MaterialTheme.colorScheme.onSurfaceVariant)
            }
            Button(onClick = onClick) { Text(buttonText) }
        }
    }
}


@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ListPreference(
    names: List<String>,
    selectedIndex: Int,
    label: String,
    onItemSelected: (Int) -> Unit
) {
    var expanded by remember { mutableStateOf(false) }
    ExposedDropdownMenuBox(
        expanded = expanded,
        onExpandedChange = { expanded = !expanded }
    ) {
        OutlinedTextField(
            value = names[selectedIndex],
            onValueChange = {},
            readOnly = true,
            label = { Text(label) },
            modifier = Modifier.menuAnchor(type = MenuAnchorType.PrimaryEditable, enabled = true).fillMaxWidth(),
            trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded = expanded) }
        )
        ExposedDropdownMenu(
            expanded = expanded,
            onDismissRequest = { expanded = false }
        ) {
            names.forEachIndexed { index, name ->
                DropdownMenuItem(
                    text = { Text(name) },
                    onClick = {
                        onItemSelected(index)
                        expanded = false
                    }
                )
            }
        }
    }
}

@Composable
fun ApiKeyInputField(
    label: String,
    apiKey: String,
    onApiKeyChange: (String) -> Unit,
    modifier: Modifier = Modifier
) {
    var passwordVisible by remember { mutableStateOf(false) }
    OutlinedTextField(
        value = apiKey,
        onValueChange = onApiKeyChange,
        label = { Text(label) },
        singleLine = true,
        visualTransformation = if (passwordVisible) VisualTransformation.None else PasswordVisualTransformation(),
        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Password),
        trailingIcon = {
            val image = if (passwordVisible) R.drawable.outline_visibility_24 else R.drawable.outline_visibility_off_24
            val description = if (passwordVisible) "Hide" else "Show"
            IconButton(onClick = { passwordVisible = !passwordVisible }) {
                Icon(painter = painterResource(image), contentDescription = description)
            }
        },
        modifier = Modifier.fillMaxWidth()
    )
}

@Composable
fun ArtworkTypeMultiSelect(
    artworkTypes: Array<String>,
    selectedTypes: Set<String>,
    onSelectionChange: (Set<String>) -> Unit
) {
    Column(Modifier.padding(16.dp)) {
        artworkTypes.forEach { type ->
            Row(
                Modifier
                    .fillMaxWidth()
                    .clickable {
                        val newSelection = selectedTypes.toMutableSet()
                        if (type in selectedTypes) newSelection.remove(type)
                        else newSelection.add(type)
                        onSelectionChange(newSelection)
                    }
                    .padding(vertical = 0.dp),
                verticalAlignment = CenterVertically
            ) {
                Checkbox(
                    checked = type in selectedTypes,
                    onCheckedChange = {
                        val newSelection = selectedTypes.toMutableSet()
                        if (it) newSelection.add(type) else newSelection.remove(type)
                        onSelectionChange(newSelection)
                    }
                )
                Spacer(Modifier.width(8.dp))
                Text(type)
            }
        }
    }
}
