package com.zell_mbc.publicartexplorer.screens

import PreferencesManager
import android.os.Build
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.AccountCircle
import androidx.compose.material.icons.filled.DateRange
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material.icons.filled.Person
import androidx.compose.material.icons.filled.PersonOff
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import com.zell_mbc.publicartexplorer.OSM_ACCESS_TOKEN_KEY
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.oauth2.TokenManager
import com.zell_mbc.publicartexplorer.osm.OSMApi
import com.zell_mbc.publicartexplorer.osm.OSMUser
import com.zell_mbc.publicartexplorer.osm.fetchChangesetCount
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.text.DateFormat
import java.time.Instant
import java.time.ZoneId
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle
import java.util.Date
import java.util.Locale

@Composable
fun OsmSection(modifier: Modifier = Modifier) {
    val context = LocalContext.current
    val prefs = PreferencesManager(context)
    val token = TokenManager.osmToken.accessToken //by prefs.getStringFlow(OSM_ACCESS_TOKEN_KEY, "").collectAsState(initial = "")

    var user by remember { mutableStateOf<OSMUser?>(null) }
    var error by remember { mutableStateOf<String?>(null) }
    var changeSets by remember { mutableStateOf<String>("0") }

    if (!token.isNullOrEmpty()) {
        LaunchedEffect(token) {
            try {
                val details = OSMApi.getUserDetails(token)
                if (details != null) {
                    user = details
                } else {
                    error = "Failed to load user info"
                }
            } catch (e: Exception) {
                error = e.message
            }
        }

        LaunchedEffect(user) {
            try {
                if (user!!.id.toString().isNotEmpty()) {
                    CoroutineScope(Dispatchers.IO).launch {
                        changeSets = fetchChangesetCount( context, token)
                        //if ()
                    }
                }
            } catch (e: Exception) {
                error = e.message
            }
        }
    }
    Surface(modifier = modifier.fillMaxWidth(), tonalElevation = 6.dp, shape = MaterialTheme.shapes.medium,) {
        Column(modifier = Modifier.padding(20.dp), verticalArrangement = Arrangement.spacedBy(16.dp)) {
            Text(
                stringResource(R.string.openstreetmap),
                style = MaterialTheme.typography.titleMedium,
                color = MaterialTheme.colorScheme.primary
            )
            if (user != null) {
                val formattedDate = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    val instant = Instant.parse(user!!.account_created)
                    val localDateTime = instant.atZone(ZoneId.systemDefault()).toLocalDateTime()

                    // Locale-aware
                    val formatter = DateTimeFormatter.ofLocalizedDateTime(FormatStyle.MEDIUM).withLocale(Locale.getDefault())
                    localDateTime.format(formatter)
                } else {
                    // Fallback for older APIs
                    val date = Date(user!!.account_created) // or parse your string using SimpleDateFormat
                    DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM, Locale.getDefault())
                        .format(date)
                }
                ItemRow(icon = Icons.Default.AccountCircle, label = stringResource(R.string.userId), value = user!!.id.toString())
                ItemRow(icon = Icons.Default.Person, label = stringResource(R.string.userName), value = user!!.display_name)
                ItemRow(icon = Icons.Default.DateRange, label = stringResource(R.string.accountCreated), value = formattedDate)
                ItemRow(icon = Icons.Default.Edit, label = stringResource(R.string.changeSets), value = changeSets)
            } else if (error != null) {
                Text("Error: $error", color = MaterialTheme.colorScheme.error)
            } else {
                if (!token.isNullOrEmpty()) ItemRow(icon = Icons.Default.PersonOff, label = stringResource(R.string.notLoggedIn), value = "")
                else Text(stringResource(R.string.loadingUserInfo),style = MaterialTheme.typography.bodyMedium, color = MaterialTheme.colorScheme.onSurfaceVariant)
            }
        }
    }
}