package com.zell_mbc.publicartexplorer.screens

import android.Manifest
import android.annotation.SuppressLint
import android.content.Context
import android.widget.Toast
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Refresh
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.SnackbarDuration
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.SnackbarResult
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Alignment.Companion.BottomEnd
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.hapticfeedback.HapticFeedbackType
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalHapticFeedback
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.core.content.edit
import com.google.accompanist.permissions.ExperimentalPermissionsApi
import com.google.accompanist.permissions.rememberMultiplePermissionsState
import com.zell_mbc.publicartexplorer.AddressInput
import com.zell_mbc.publicartexplorer.Artwork
import com.zell_mbc.publicartexplorer.BuildConfig
import com.zell_mbc.publicartexplorer.Constants.DEFAULT_ZOOM
import com.zell_mbc.publicartexplorer.DebugLog
import com.zell_mbc.publicartexplorer.LATITUDE_KEY
import com.zell_mbc.publicartexplorer.LONGITUDE_KEY
import com.zell_mbc.publicartexplorer.MAPBOX_TILE_SERVER
import com.zell_mbc.publicartexplorer.MAPTOOLKIT_API_KEY
import com.zell_mbc.publicartexplorer.MAPTOOLKIT_TILE_SERVER
import com.zell_mbc.publicartexplorer.NodeType
import com.zell_mbc.publicartexplorer.NodeType.ARCHITECTURE
import com.zell_mbc.publicartexplorer.NodeType.BUST
import com.zell_mbc.publicartexplorer.NodeType.FOUNTAIN
import com.zell_mbc.publicartexplorer.NodeType.GRAFFITI
import com.zell_mbc.publicartexplorer.NodeType.INSTALLATION
import com.zell_mbc.publicartexplorer.NodeType.MURAL
import com.zell_mbc.publicartexplorer.NodeType.MUSEUM
import com.zell_mbc.publicartexplorer.NodeType.RELIEF
import com.zell_mbc.publicartexplorer.NodeType.SCULPTURE
import com.zell_mbc.publicartexplorer.NodeType.STATUE
import com.zell_mbc.publicartexplorer.NodeType.STONE
import com.zell_mbc.publicartexplorer.MIN_MOVEMENT_METERS
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.ZOOM_KEY
import com.zell_mbc.publicartexplorer.maps.createCircledIcon
import com.zell_mbc.publicartexplorer.data.ViewModel
import com.zell_mbc.publicartexplorer.detailview.EditTagsDialog
import com.zell_mbc.publicartexplorer.detailview.ShowDetails
import com.zell_mbc.publicartexplorer.detailview.ShowRawData
import com.zell_mbc.publicartexplorer.maps.getLocation
import com.zell_mbc.publicartexplorer.maps.calculateAnimationDuration
import com.zell_mbc.publicartexplorer.maps.distanceInMeters
import com.zell_mbc.publicartexplorer.maps.getLastKnownLocation
import com.zell_mbc.publicartexplorer.maps.getMapToolkitUrl
import com.zell_mbc.publicartexplorer.maps.getMapboxUrl
import com.zell_mbc.publicartexplorer.maps.getOpenFreeMapUrl
import com.zell_mbc.publicartexplorer.oauth2.startOsmLogin
import dev.sargunv.maplibrecompose.compose.MaplibreMap
import dev.sargunv.maplibrecompose.compose.layer.SymbolLayer
import dev.sargunv.maplibrecompose.compose.rememberCameraState
import dev.sargunv.maplibrecompose.compose.source.rememberGeoJsonSource
import dev.sargunv.maplibrecompose.core.CameraPosition
import io.github.dellisd.spatialk.geojson.Position
import kotlinx.coroutines.launch
import dev.sargunv.maplibrecompose.compose.ClickResult
import dev.sargunv.maplibrecompose.compose.layer.CircleLayer
import dev.sargunv.maplibrecompose.compose.rememberStyleState
import dev.sargunv.maplibrecompose.core.BaseStyle
import dev.sargunv.maplibrecompose.core.MapOptions
import dev.sargunv.maplibrecompose.core.OrnamentOptions
import dev.sargunv.maplibrecompose.core.source.GeoJsonData
import dev.sargunv.maplibrecompose.core.source.GeoJsonOptions
import dev.sargunv.maplibrecompose.expressions.dsl.asString
import dev.sargunv.maplibrecompose.expressions.dsl.case
import dev.sargunv.maplibrecompose.expressions.dsl.const
import dev.sargunv.maplibrecompose.expressions.dsl.feature
import dev.sargunv.maplibrecompose.expressions.dsl.image
import dev.sargunv.maplibrecompose.expressions.dsl.switch
import io.github.dellisd.spatialk.geojson.Point
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import org.maplibre.android.geometry.LatLng
import kotlin.time.Duration.Companion.milliseconds
import LocationTracker
import androidx.compose.ui.Alignment.Companion.BottomStart
import com.zell_mbc.publicartexplorer.MAPBOX_API_KEY
import com.zell_mbc.publicartexplorer.maps.MapAttributionOverlay
import com.zell_mbc.publicartexplorer.oauth2.TokenManager
import org.json.JSONObject

@OptIn(ExperimentalPermissionsApi::class)
@SuppressLint("DefaultLocale")
@Composable
fun MapScreen(snackbarHostState: SnackbarHostState, viewModel: ViewModel) {
    val context = LocalContext.current
    val preferences = context.getSharedPreferences(
        "com.zell_mbc.publicartexplorer._preferences",
        Context.MODE_PRIVATE
    )

    //preferences.edit { putBoolean(SUBSCRIPTION_ACTIVE_KEY, false) } // Debug, delete when done
    val scope = rememberCoroutineScope()
    val artworkGeoJsonData by viewModel.artworkGeoJsonData.collectAsState()

    val storedLongitude = Double.fromBits(preferences.getLong(LONGITUDE_KEY, 0L))
    val storedLatitude = Double.fromBits(preferences.getLong(LATITUDE_KEY, 0L))
    val storedZoom = Double.fromBits(preferences.getLong(ZOOM_KEY, 0L))

    // Create icons
    val iconBackground = MaterialTheme.colorScheme.tertiaryContainer.toArgb()
    val iconForeground = MaterialTheme.colorScheme.onTertiaryContainer.toArgb()
    val iconBust = remember {
        createCircledIcon(
            context,
            R.drawable.outline_person_24,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconMuseum = remember {
        createCircledIcon(
            context,
            R.drawable.outline_museum_24,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconStatue = remember {
        createCircledIcon(
            context,
            R.drawable.icon_statue,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconSculpture = remember {
        createCircledIcon(
            context,
            R.drawable.icon_artwork,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconGraffiti = remember {
        createCircledIcon(
            context,
            R.drawable.outline_fragrance_24,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconMural = remember {
        createCircledIcon(
            context,
            R.drawable.outline_imagesmode_24,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconRelief = remember {
        createCircledIcon(
            context,
            R.drawable.outline_perm_media_24,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconInstallation = remember {
        createCircledIcon(
            context,
            R.drawable.icon_installation,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconUnknown = remember {
        createCircledIcon(
            context,
            R.drawable.outline_question_mark_24,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconFountain = remember {
        createCircledIcon(
            context,
            R.drawable.outline_sprinkler_24,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconArchitecture = remember {
        createCircledIcon(
            context,
            R.drawable.outline_architecture_24,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val iconStone = remember {
        createCircledIcon(
            context,
            R.drawable.icon_stone,
            backgroundColor = iconBackground,
            foregroundColor = iconForeground
        ).asImageBitmap()
    }
    val allIcons = listOf(
        iconBust,
        iconMuseum,
        iconStatue,
        iconSculpture,
        iconGraffiti,
        iconMural,
        iconRelief,
        iconInstallation,
        iconUnknown,
        iconFountain,
        iconArchitecture
    )
    // Call prepareToDraw() on ImageBitmaps for GPU upload optimization (optional but recommended)
    LaunchedEffect(allIcons) {
        allIcons.forEach { it.prepareToDraw() }
    }

    val coroutineScope = rememberCoroutineScope()
    val styleState = rememberStyleState()

    val mapOptions = MapOptions(
        ornamentOptions = OrnamentOptions(
            isScaleBarEnabled = true,
            isCompassEnabled = false,
            isAttributionEnabled = false,
            attributionAlignment = BottomStart,
            isLogoEnabled = false
        )
    )

    // Longitude and latitude are set -> we can set the camera position to whatever's stored in preferences
    val cameraState = rememberCameraState(
        firstPosition = CameraPosition(target = Position(latitude = storedLatitude, longitude = storedLongitude), zoom = storedZoom)
    )

    val permissionsState = rememberMultiplePermissionsState(
        permissions = listOf(
            Manifest.permission.ACCESS_FINE_LOCATION,
            Manifest.permission.ACCESS_COARSE_LOCATION
        )
    )

    // 1. Define the permissions I want. Request both fine and coarse location permissions
    LaunchedEffect(Unit) {// 2. Request the permissions.
        if (!permissionsState.allPermissionsGranted) {
            permissionsState.launchMultiplePermissionRequest()
        }
    }

    LaunchedEffect(permissionsState.allPermissionsGranted) {
        // Make sure the "dot" is drawn right at launch
        val lastKnown = getLastKnownLocation(context)
        if (lastKnown != null) {
            viewModel.userLocation = Position(lastKnown.longitude, lastKnown.latitude)
            val locationAgeMillis = System.currentTimeMillis() - lastKnown.time
            val locationAgeSeconds = locationAgeMillis / 1000
            viewModel.gpsFixAvailable = (locationAgeSeconds < 120) // consider everything higher than 2 minutes as stale
        }

        if (cameraState.position.target.longitude + cameraState.position.target.latitude == 0.0) { // No stored location = First app start: Shared pref's do not contain a lastknownLocation
            if (lastKnown != null) {
                coroutineScope.launch {
                    cameraState.animateTo(
                        duration = 2000.milliseconds,
                        finalPosition = CameraPosition(target = Position(latitude = lastKnown.latitude, longitude = lastKnown.longitude), zoom = DEFAULT_ZOOM, bearing = 0.0, tilt = 0.0)
                    )
                }
                val locationAgeMillis = System.currentTimeMillis() - lastKnown.time
                val locationAgeSeconds = locationAgeMillis / 1000
                viewModel.gpsFixAvailable = (locationAgeSeconds < 120) // consider everything higher than 2 minutes as stale -> gray dow instead of blue
            }
            else { // Try to retrieve location
                getLocation(context, callback = {
                        if (it != null) {
                            viewModel.gpsFixAvailable = true
                            DebugLog.add("User location from getLocation, " + String.format("%.3f", it.latitude) + "," + String.format("%.3f", it.longitude))
                            viewModel.userLocation = Position(it.longitude, it.latitude)
                            coroutineScope.launch {
                                cameraState.animateTo(
                                    duration = 2000.milliseconds,
                                    finalPosition = CameraPosition(target = Position(latitude = it.latitude, longitude = it.longitude), zoom = DEFAULT_ZOOM, bearing = 0.0, tilt = 0.0)
                                )
                            }
                        }
                        // else = Unable to retrieve location by any means, stay on 0,0 is the best we can do
                })
            }
        }
    }

    // Have to map follow the user
    val tracker = remember { LocationTracker(context) }
    LaunchedEffect(viewModel.followMe) {
        if (viewModel.followMe) {
            tracker.startTracking { newLocation ->
                coroutineScope.launch {
                    val currentLatLng = LatLng(
                        latitude = cameraState.position.target.latitude,
                        longitude = cameraState.position.target.longitude
                    )
                    val targetLatLng = LatLng(newLocation.latitude, newLocation.longitude)

                    // Only animate if user moved enough
                    if (distanceInMeters(currentLatLng, targetLatLng) >= MIN_MOVEMENT_METERS) {
                        if (viewModel.debugMode) {
                            DebugLog.add("Camera: " + String.format("%.4f", currentLatLng.latitude) + ", " + String.format("%.4f", currentLatLng.longitude))
                            DebugLog.add("Target: " + String.format("%.4f", targetLatLng.latitude) + ", " + String.format("%.4f", targetLatLng.longitude))
                            DebugLog.add("Delta: " + String.format("%.4f", targetLatLng.latitude - currentLatLng.latitude) +", " + String.format("%.4f", targetLatLng.longitude - currentLatLng.longitude)
                            )
                        }

                        cameraState.animateTo(
                            duration = calculateAnimationDuration(currentLatLng, targetLatLng).milliseconds,
                            finalPosition = CameraPosition(target = Position(longitude = newLocation.longitude, latitude = newLocation.latitude),
                                zoom = cameraState.position.zoom, bearing = 0.0, tilt = 0.0)
                        )

                        viewModel.userLocation = Position(newLocation.longitude, newLocation.latitude)
                    }
                }
            }
        } else tracker.stopTracking()
    }

    // This block runs whenever the camera's center/zoom/bearing/pitch changes.
    var saveJob by remember { mutableStateOf<Job?>(null) }
    LaunchedEffect(cameraState.position) {
        // Cancel any previous scheduled save when the position changes
        saveJob?.cancel()

        // Launch a new save job with delay (debounce)
        saveJob = launch {
            delay(2000L)  // debounce period: 2 seconds of no position change

            val pos = cameraState.position
            if (pos.target.latitude == 0.0 && pos.target.longitude == 0.0) return@launch // ignore invalid

            // Save the stable position after debounce
            //viewModel.cameraLatitude  = pos.target.latitude
            //viewModel.cameraLongitude = pos.target.longitude
            //viewModel.zoom      = pos.zoom

            preferences.edit {
                putLong(LATITUDE_KEY, pos.target.latitude.toRawBits())
                putLong(LONGITUDE_KEY, pos.target.longitude.toRawBits())
                putLong(ZOOM_KEY, pos.zoom.toRawBits())
            }
            //Log.d("Position", "Saved ${pos.target.latitude},${pos.target.longitude}, ${pos.zoom}")
            DebugLog.add(
                "Position saved," +
                        String.format("%.3f", pos.target.latitude) + "," +
                        String.format("%.3f", pos.target.longitude) + "," +
                        String.format("%.3f", pos.zoom)
            )
        }
    }

    // Build the full URL string
    val selectedStyle by remember(
        viewModel.selectedTileServer,
        viewModel.selectedMapStyleIndex) {
        mutableStateOf(
            when (viewModel.selectedTileServer) {
                MAPBOX_TILE_SERVER -> {
                    val apiKey = BuildConfig.MAPBOX_API_KEY.ifEmpty { preferences.getString(MAPBOX_API_KEY, "") ?: "" }
                    if (apiKey.isEmpty()) {
                        Toast.makeText(context, context.getString(R.string.emptyApiKey), Toast.LENGTH_LONG).show()
                    }
                    // Returns the JSON string for MapLibre
                    getMapboxUrl(viewModel.selectedMapStyleIndex, apiKey, context)
                }
                MAPTOOLKIT_TILE_SERVER -> {
                    val apiKey = BuildConfig.MAPTOOLKIT_API_KEY.ifEmpty { preferences.getString(MAPTOOLKIT_API_KEY, "") ?: "" }
                    if (apiKey.isEmpty()) {
                        Toast.makeText(context, context.getString(R.string.emptyApiKey), Toast.LENGTH_LONG).show()
                    }
                    getMapToolkitUrl(viewModel.selectedMapStyleIndex, apiKey)
                }
                else -> getOpenFreeMapUrl(viewModel.selectedMapStyleIndex)
            }
        )
    }

    //Mapbox: selectedStyle = "file:///android_asset/light-v10.json"
    // OSM:
    //Log.d("MapStyle", selectedStyle)

    Box(modifier = Modifier.fillMaxSize()) {
        MaplibreMap(
            baseStyle = if (viewModel.selectedTileServer == MAPBOX_TILE_SERVER) BaseStyle.Json(selectedStyle) else BaseStyle.Uri(selectedStyle),
            cameraState = cameraState,
            options = mapOptions,
            styleState = styleState,
            onMapLoadFinished = { /* handle loaded */ },
            onMapLongClick = { pos, offset ->
                if (viewModel.expertMode) {
                    if (TokenManager.osmToken.accessToken.isNullOrEmpty())
                        scope.launch {
                            val result = snackbarHostState.showSnackbar(
                                message = context.getString(R.string.osmNotSignedIn),
                                actionLabel = context.getString(R.string.login),
                                duration = SnackbarDuration.Short
                            )
                            when (result) {
                                SnackbarResult.ActionPerformed -> startOsmLogin(context)
                                SnackbarResult.Dismissed -> { /* Do nothing */
                                }
                            }
                        }
                    else {
                        viewModel.selectedPosition = pos
                        viewModel.showNewNodeDialog = true
                    }
                }
                ClickResult.Pass
            }
        ) {
            artworkGeoJsonData.let { json ->
                val source = rememberGeoJsonSource(
                    data = GeoJsonData.JsonString(json.toString()),
                    options = GeoJsonOptions()
                )

                SymbolLayer(
                    id = "artwork-layer",
                    source = source,
                    iconImage = switch(
                        input = feature["artwork_type"].asString(),
                        case(label = SCULPTURE.toString(), output = image(iconSculpture)),
                        case(label = STATUE.toString(), output = image(iconStatue)),
                        case(label = MUSEUM.toString(), output = image(iconMuseum)),
                        case(label = GRAFFITI.toString(), output = image(iconGraffiti)),
                        case(label = MURAL.toString(), output = image(iconMural)),
                        case(label = RELIEF.toString(), output = image(iconRelief)),
                        case(label = INSTALLATION.toString(), output = image(iconInstallation)),
                        case(label = FOUNTAIN.toString(), output = image(iconFountain)),
                        case(label = ARCHITECTURE.toString(), output = image(iconArchitecture)),
                        case(label = BUST.toString(), output = image(iconBust)),
                        case(label = STONE.toString(), output = image(iconStone)),
                        fallback = image(iconUnknown)
                    ),
                    //iconAllowOverlap = const(false),         // ✅ show all icons even if they overlap
                    iconIgnorePlacement = const(true), //viewModel.zoomLevelFiltering),      // ✅ do not check collision
                    minZoom = 0f,
                    onClick = { features ->
                        val arrayId = features.firstOrNull()?.getStringProperty("array_id")
                        if (arrayId != null) {
                            try {
                                viewModel.selectArtwork(arrayId.toInt()) // = viewModel.artworks[arrayIdValue]
                            } catch (_: Exception) {
                                Toast.makeText(context, context.getString(R.string.unableToLoadArtwork), Toast.LENGTH_LONG).show()
                            }
                        }
                        ClickResult.Consume
                    }
                )
            }

            // Show small circle for user location
            viewModel.userLocation?.let { position ->
                val userPositionSource = rememberGeoJsonSource(data = GeoJsonData.Features(Point(position)))
                /* Blue location dot fill */
                val circleColor = if (viewModel.gpsFixAvailable) Color(4, 151, 255, 255) else Color(98, 196, 161, 255) //* translucent light gray */
                CircleLayer(
                    id = "user-location-layer",
                    source = userPositionSource,
                    color = const(circleColor), //Color(66, 133, 244, 100)),
                    radius = const(10.dp),
                    strokeColor = const(Color(255, 255, 255, 255)),          // white stroke color
                    strokeWidth = const(3.dp)                   // stroke width in pixels (adjust as needed)
                )
            }

            // Artwork selected
            if (viewModel.selectedArtwork != null) {
                var showHelp by remember { mutableStateOf(false) }
                var helpMessages by remember { mutableStateOf(listOf("")) }

                // Make sure fecthImages get's only called when selectedArtwork changes and not during recompositions
                LaunchedEffect(viewModel.selectedArtwork) {
                    viewModel.fetchImages()
                }

                AlertDialog(
                    onDismissRequest = { /* This is here to avoid closing the dialog by clicking outside */ },
                    confirmButton = {
                        Button(onClick = { viewModel.selectedArtwork = null }) {
                            Text(stringResource(R.string.done))
                        }
                    },
                    dismissButton = if (viewModel.expertMode) {
                        {
                            OutlinedButton(onClick = {
                                viewModel.showRawData = !viewModel.showRawData
                            }) {
                                Text(if (viewModel.showRawData) stringResource(R.string.details) else stringResource(R.string.rawData))
                            }
                        }
                    } else null,
                    title = {
                        Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.SpaceBetween, verticalAlignment = Alignment.CenterVertically) {
                            Text(if (!viewModel.showRawData) stringResource(R.string.details) + ":" else stringResource(R.string.rawData) + ":")
                            Icon(painter = painterResource(R.drawable.outline_help_24), contentDescription = "Help",
                                modifier = Modifier.size(24.dp)
                                    .clickable {
                                        helpMessages = listOf(
                                            context.getString(R.string.helpDataMissing1),
                                            context.getString(R.string.helpDataMissing2),
                                            context.getString(R.string.helpDataMissing3),
                                            context.getString(R.string.helpDataMissing4)
                                        )
                                        showHelp = true
                                    })
                        }
                    },
                    text = {
                        viewModel.selectedArtwork?.let { artwork ->
                            if (viewModel.showRawData) ShowRawData(viewModel, artwork)
                            else                       ShowDetails(viewModel, artwork)
                        }
                    }
                )

                // Overlay Help AlertDialog
                if (showHelp) {
                    AlertDialog(
                        onDismissRequest = { showHelp = false },
                        title = { Text("Help") },
                        text = {
                            Column(modifier = Modifier.verticalScroll(rememberScrollState())) {
                                helpMessages.forEach { line ->
                                    Text(line)
                                    Spacer(modifier = Modifier.height(4.dp)) // spacing between lines
                                }
                            }
                        },
                        confirmButton = {
                            OutlinedButton(onClick = { showHelp = false }) {
                                Text(stringResource(R.string.ok))
                            }
                        }
                    )
                }
            }

            if (viewModel.showSearchCircle) {
                val pos = cameraState.position
                val circle = rememberGeoJsonSource(
                    data = GeoJsonData.Features(
                        Point(
                            Position(
                                pos.target.longitude,
                                pos.target.latitude
                            )
                        )
                    ),
                )
                //Log.d("Position", "Show circle ${viewModel.latitude},${viewModel.longitude}, ${viewModel.zoom}")
                DebugLog.add("Circle center: ${pos.target.latitude},${pos.target.longitude}")
                val radiusInDPAtTarget = viewModel.radius / cameraState.metersPerDpAtTarget
                CircleLayer(
                    id = "circle-layer",
                    source = circle,
                    color = const(Color(0, 0, 255, 50)),
                    radius = const(radiusInDPAtTarget.dp)
                )
            }

            if (viewModel.showAddressSearch) {
                AddressInput(
                    onSearchComplete = { lat, lon ->
                        if (lat + lon == 0.0) viewModel.showAddressSearch = false
                        else {
                            viewModel.followMe = false // Make sure followMe is off as otherwise the map will go back to where the user is

                            // Save the lat/lon returned from the dialog
                            coroutineScope.launch {
                                val currentLatLng = LatLng(
                                    latitude = cameraState.position.target.latitude,
                                    longitude = cameraState.position.target.longitude
                                )
                                val targetLatLng = LatLng(lat, lon)
                                cameraState.animateTo(
                                    duration = calculateAnimationDuration(currentLatLng, targetLatLng).milliseconds,
                                    finalPosition = CameraPosition(target = Position(longitude = lon, latitude = lat), zoom = DEFAULT_ZOOM, bearing = 0.0, tilt = 0.0)
                                )
                            }
                            DebugLog.add("Search, pos = " + String.format("%.3f", lat) + "," + String.format("%.3f", lon))
                            viewModel.showAddressSearch = false
                        }
                    }
                )
            }

            if (viewModel.showNewNodeDialog) {
                if (viewModel.selectedPosition != null) {
                    val initialTags: MutableMap<String, String> = mutableMapOf()
                    // Add default list of artwork tags
                    initialTags["name"] = ""
                    initialTags["artist_name"] = ""
                    initialTags["artwork_type"] = ""
                    initialTags["material"] = ""
                    initialTags["wikimedia_commons"] = ""
                    initialTags["description"] = ""
                    //initialTags["image"] = ""
                    initialTags["wheelchair"] = ""
                    //initialTags["website"] = ""

                    val newArtwork = Artwork(
                        id = "",
                        viewModel.selectedPosition!!.latitude,
                        viewModel.selectedPosition!!.longitude,
                        initialTags,
                        //NodeType.UNKNOWN
                    )

                    EditTagsDialog(viewModel,
                        stringResource(R.string.newArtwork),
                        tags = newArtwork.tags,
                        onDismiss = { viewModel.showNewNodeDialog = false },
                        onDialogConfirm = {
                            viewModel.showNewNodeDialog = false
                            if (!it.values.all { it.isBlank() }) {
                                newArtwork.tags = it
                                newArtwork.tags["tourism"] = "artwork" // Mandatory tag! Make sure it's set. Without this it won't be picked up as artwork
                                newArtwork.tags.entries.removeIf { (_, value) -> value.isBlank() } // Remove empty tags
                                newArtwork.tags = LinkedHashMap(it) // Convert from state to actual value to make sure tags survive a scope change
                                // Get numerical value for node type
                               /* val artworkType = newArtwork.tags["artwork_type"]
                                if (artworkType != null) {
                                    val tmpJson = JSONObject("""{"artwork_type":"${artworkType}"}""")
                                    newArtwork.type = viewModel.getNodeType(tmpJson)
                                } */
                                // Add new node to existing data set and reload images in case one of the image tags got changed
                                viewModel.artworks.add(newArtwork)
                                viewModel.fetchImages()
                                viewModel.updateOsm(newArtwork, TokenManager.osmToken.accessToken ?: "") // Create new node on Osm, NodeId will be update once available

                                // Need to build a new geoString from updated array
                                coroutineScope.launch {
                                    viewModel.setArtworkGeoJsonData(viewModel.overpassToGeoJson())
                                }
                            }
                        }
                    )
                }
            }
        }

        FabMenu(viewModel, (viewModel.expertMode && viewModel.artworkCount.value > 0), findMe = {
            viewModel.findMe = true
            val zoom = cameraState.position.zoom

            // Try lastknown first
            val pos = getLastKnownLocation(context)
            if (pos != null)
                coroutineScope.launch {
                    cameraState.animateTo(duration = 2000.milliseconds, finalPosition = CameraPosition(target = Position(latitude = pos.latitude, longitude = pos.longitude), zoom = zoom, bearing = 0.0, tilt = 0.0))
                }
            getLocation(context, callback = {
                if (it != null) {
                    viewModel.userLocation = Position(it.longitude, it.latitude)
                    viewModel.findMe = false
                    coroutineScope.launch {
                        cameraState.animateTo(duration = 2000.milliseconds,
                            finalPosition = CameraPosition(target = Position(latitude = it.latitude, longitude = it.longitude), zoom = zoom))
                    }
                }
                else viewModel.findMe = false // Timeout?
            })
        })

        val hapticFeedback = LocalHapticFeedback.current
        FloatingActionButton(
            modifier = Modifier.padding(end = 16.dp, bottom = 16.dp).align(BottomEnd),
            onClick = {
                if (viewModel.enableHapticFeedback) hapticFeedback.performHapticFeedback(
                    HapticFeedbackType.Confirm
                )
                coroutineScope.launch {
                    Toast.makeText(
                        context,
                        context.getString(R.string.loadingArtworks, viewModel.radius),
                        Toast.LENGTH_LONG
                    ).show()
                    val pos = cameraState.position
                    DebugLog.add("Search center/cameraPosition lat :" + pos.target.latitude.toString() + " lon: " + pos.target.longitude.toString())
                    DebugLog.add("userLocation: ${viewModel.userLocation}")
                    val geoString = viewModel.fetchArtworks(
                        context = context,
                        latitude = pos.target.latitude,
                        longitude = pos.target.longitude
                    )
                    viewModel.setArtworkGeoJsonData(geoString)
                }
            }
        ) { Icon(Icons.Filled.Refresh, contentDescription = "Load Artwork") }

        MapAttributionOverlay(selectedTileServer = viewModel.selectedTileServer)

    }
}