package com.zell_mbc.publicartexplorer.screens

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.location.LocationManager
import android.os.Build
import android.widget.Toast
import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.*
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.core.content.edit
import com.google.accompanist.permissions.ExperimentalPermissionsApi
import com.google.accompanist.permissions.rememberMultiplePermissionsState
import com.zell_mbc.publicartexplorer.BuildConfig
import com.zell_mbc.publicartexplorer.DEBUG_MODE_KEY
import com.zell_mbc.publicartexplorer.DebugLog
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.data.ViewModel
import com.zell_mbc.publicartexplorer.maps.getLastKnownLocation
import com.zell_mbc.publicartexplorer.maps.locationEnabled
import com.zell_mbc.publicartexplorer.onboarding.BUILD_KEY
import com.zell_mbc.publicartexplorer.appVersion
import java.text.DateFormat
import java.util.*


@OptIn(ExperimentalPermissionsApi::class)
@Composable
fun InfoScreen(viewModel: ViewModel) {
    val context = LocalContext.current
    val preferences = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
    val scrollState = rememberScrollState()
    val packageInfo = if (Build.VERSION.SDK_INT >= 33) { context.packageManager.getPackageInfo(context.packageName,         PackageManager.PackageInfoFlags.of(0))
    } else {
        @Suppress("DEPRECATION")
        context.packageManager.getPackageInfo(context.packageName, 0)
    }

    val versionCode = preferences.getInt(BUILD_KEY,0).toString()
    val versionName = buildString {
        append(appVersion)
        append(if (BuildConfig.FLAVOUR == "foss") " FOSS" else " Play")
    }

    // Permissions state for location
    val permissionsState = rememberMultiplePermissionsState(
        permissions = listOf(
            Manifest.permission.ACCESS_FINE_LOCATION,
            Manifest.permission.ACCESS_COARSE_LOCATION
        )
    )
    val permissionsGranted = permissionsState.allPermissionsGranted

    val lastKnownLocation = getLastKnownLocation(context)
    val locationManager = context.getSystemService(Context.LOCATION_SERVICE) as LocationManager
    val enabledProviders = locationManager.getProviders(true)
    val locationProvider = lastKnownLocation?.provider ?: stringResource(R.string.unknown) //enabledProviders.toString()

    // Format last location time nicely, fallback to "unknown"
    val locationTime = lastKnownLocation?.let { location ->
        val dateFormatter = DateFormat.getDateTimeInstance() // Default style, local format
        dateFormatter.format(Date(location.time))
    } ?: stringResource(R.string.unknown)

    //val mapProvider = tileServers[preferences.getInt(TILE_SERVER_KEY, 0)].name
    Column(modifier = Modifier.fillMaxSize().safeDrawingPadding().verticalScroll(scrollState).padding(horizontal = 16.dp, vertical = 12.dp), verticalArrangement = Arrangement.spacedBy(16.dp)) {
        // Logo with hidden tap setting
        val s = stringResource(R.string.debugMode) + "  " + if (!viewModel.debugMode) stringResource(R.string.enabled) else stringResource(R.string.disabled)
        LogoWithTapSetting(
            modifier = Modifier.align(Alignment.CenterHorizontally),
            onFiveTaps = {
                viewModel.debugMode = !viewModel.debugMode
                DebugLog.add("Lastknown location, lat: " + lastKnownLocation?.latitude + " ,lon: " + lastKnownLocation?.longitude)
                preferences.edit { putBoolean(DEBUG_MODE_KEY, viewModel.debugMode) } // no apply needed inside edit
                Toast.makeText(context, s, Toast.LENGTH_LONG).show()
            }
        )

        Surface(
            modifier = Modifier.fillMaxWidth(),
            tonalElevation = 6.dp,
            shape = MaterialTheme.shapes.medium
        ) {
            Column(
                modifier = Modifier.padding(16.dp),
                verticalArrangement = Arrangement.spacedBy(8.dp)
            ) {
                Text(text = stringResource(id = R.string.version), style = MaterialTheme.typography.titleMedium, color = MaterialTheme.colorScheme.primary)
                Text(text = "$versionName, Build: $versionCode", style = MaterialTheme.typography.bodyMedium)
                //Text(text = stringResource(id = R.string.mapProvider) + ": $mapProvider", style = MaterialTheme.typography.bodyMedium)
                if (BuildConfig.FLAVOUR == "play") {
                    val s = if (viewModel.subscriptionActive) {
                        val dateFormatter = DateFormat.getDateInstance() // Default style, local format
                        val formattedDate = dateFormatter.format(Date(viewModel.expiryTimestamp))
                        stringResource(R.string.subscriptionActiveUntil) + " $formattedDate"
                    } else stringResource(R.string.subscriptionInactive)
                Text(text = s, style = MaterialTheme.typography.bodyMedium)
                }
            }
        }

        ContactSection()
        ConnectionSection(locationEnabled = locationEnabled(context), locationProvider = locationProvider, locationTime = locationTime, permissionsGranted = permissionsGranted)
        if (viewModel.expertMode) {
            OsmSection()
            WikiSection()
        }
        ProjectInfoSection()

    }
}

@Composable
fun LogoWithTapSetting(
    modifier: Modifier = Modifier,
    onFiveTaps: () -> Unit
) {
    var tapCount by remember { mutableIntStateOf(0) }

    Image(
        painter = painterResource(id = if (isSystemInDarkTheme()) R.drawable.featuregraphic_dark else R.drawable.featuregraphic_light),
        contentDescription = stringResource(id = R.string.appName),
        modifier = modifier
            .size(200.dp)
            .clickable {
                tapCount++
                if (tapCount == 5) {
                    onFiveTaps()
                    tapCount = 0
                }
            }
    )
}
