package com.zell_mbc.publicartexplorer.screens

import android.widget.Toast
import androidx.compose.animation.*
import androidx.compose.foundation.layout.*
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Cancel
import androidx.compose.material.icons.filled.LocationOff
import androidx.compose.material.icons.filled.LocationOn
import androidx.compose.material.icons.filled.MyLocation
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.hapticfeedback.HapticFeedbackType
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalHapticFeedback
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import com.zell_mbc.publicartexplorer.data.ViewModel
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.downloadArtworkList
import com.zell_mbc.publicartexplorer.isNetworkAvailable

@Composable
fun FabMenu(viewModel: ViewModel, showDownloadFab: Boolean, findMe: () -> Unit) {
    val context = LocalContext.current
    val hapticFeedback = LocalHapticFeedback.current

    var expanded by remember { mutableStateOf(false) }

    Box(modifier = Modifier.fillMaxSize(), contentAlignment = Alignment.TopEnd) {
        // --- All sub-FABs wrapped together ---
        AnimatedVisibility(visible = expanded, enter = fadeIn() + expandVertically(expandFrom = Alignment.Top), exit = fadeOut() + shrinkVertically(shrinkTowards = Alignment.Top)) {
            Column(horizontalAlignment = Alignment.End, verticalArrangement = Arrangement.spacedBy(12.dp),
                modifier = Modifier.padding(end = 16.dp, top = 88.dp) // space for main FAB
            ) {
                // Show search circle
                ExtendedFloatingActionButton(
                    containerColor = MaterialTheme.colorScheme.secondaryContainer,
                    contentColor = MaterialTheme.colorScheme.onSecondaryContainer,
                    text = { Text(stringResource(R.string.showSearchCircle)) },
                    icon = { if (!viewModel.showSearchCircle) Icon(painter = painterResource(id = R.drawable.outline_show_circle_24), contentDescription = stringResource(R.string.showSearchCircle))
                        else Icon(painter = painterResource(id = R.drawable.outline_circle_24), contentDescription = stringResource(R.string.showSearchCircle))
                    },
                    onClick = {
                        viewModel.showSearchCircle = !viewModel.showSearchCircle
                        expanded = false
                    }
                )
                // FindMe
                ExtendedFloatingActionButton(
                    containerColor = MaterialTheme.colorScheme.secondaryContainer,
                    contentColor = MaterialTheme.colorScheme.onSecondaryContainer,
                    text = { Text(stringResource(R.string.findMe)) },
                    icon = { Icon(Icons.Filled.MyLocation, contentDescription = stringResource(R.string.findMe)) },
                    onClick = {
                        findMe()
                        expanded = false
                    }
                )

                // FollowMe
                ExtendedFloatingActionButton(
                    containerColor = MaterialTheme.colorScheme.secondaryContainer,
                    contentColor = MaterialTheme.colorScheme.onSecondaryContainer,
                    text = { Text(stringResource(R.string.followMe)) },
                    icon = { if (viewModel.followMe) Icon(Icons.Default.LocationOn, contentDescription = stringResource(R.string.followMe))
                    else Icon(Icons.Default.LocationOff, contentDescription = null) },
                    onClick = {
                        viewModel.followMe = !viewModel.followMe
                        expanded = false
                    }
                )

                // Address search
                ExtendedFloatingActionButton(
                    containerColor = MaterialTheme.colorScheme.secondaryContainer,
                    contentColor = MaterialTheme.colorScheme.onSecondaryContainer,
                    text = { Text(stringResource(R.string.search)) },
                    icon = { Icon(painter = painterResource(id = R.drawable.outline_search_24), contentDescription = stringResource(R.string.search)) },
                    onClick = {
                        if (viewModel.enableHapticFeedback) hapticFeedback.performHapticFeedback(HapticFeedbackType.Confirm)
                        // Check Network connection
                        if (!isNetworkAvailable(context)) {
                            Toast.makeText(context, context.getString(R.string.deviceOffline), Toast.LENGTH_LONG).show()
                        } else viewModel.showAddressSearch = true
                        expanded = false
                    }
                )
                if (showDownloadFab) {
                    // Show download
                    ExtendedFloatingActionButton(
                        containerColor = MaterialTheme.colorScheme.secondaryContainer,
                        contentColor = MaterialTheme.colorScheme.onSecondaryContainer,
                        text = { Text(stringResource(R.string.download)) },
                        icon = { Icon(painter = painterResource(id = R.drawable.outline_download_24), contentDescription = stringResource(R.string.download)) },
                        onClick = {
                            expanded = false
                            if (viewModel.enableHapticFeedback) hapticFeedback.performHapticFeedback(HapticFeedbackType.Confirm)
                            downloadArtworkList(context, viewModel)
                        }
                    )
                 }
            }
        }

        // --- Main FAB ---
        ExtendedFloatingActionButton(
            text = { Text(if (expanded) stringResource(R.string.close) else stringResource(R.string.menu)) },
            containerColor = MaterialTheme.colorScheme.primaryContainer,
            contentColor = MaterialTheme.colorScheme.onPrimaryContainer,
            icon = { if (!expanded) Icon(Icons.Default.Add, contentDescription = stringResource(R.string.menu))
                    else Icon(Icons.Default.Cancel, contentDescription = stringResource(R.string.menu))
                   },
            onClick = { expanded = !expanded },
            modifier = Modifier.padding(end = 16.dp, top = 16.dp)
        )

        if (viewModel.findMe) {
            Row(modifier = Modifier.fillMaxWidth().padding(end = 16.dp, bottom = 8.dp), horizontalArrangement = Arrangement.End) {
                CircularProgressIndicator(color = MaterialTheme.colorScheme.primary, modifier = Modifier.size(32.dp))
            }
        }
    }
}
