package com.zell_mbc.publicartexplorer.onboarding

import android.content.Context
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.navigationBarsPadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.rememberPagerState
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.zell_mbc.publicartexplorer.R
import kotlinx.coroutines.launch

@Composable
fun OnboardingPage(modifier: Modifier = Modifier, context: Context, setValue: (AppStart) -> Unit) {
    val fg = if (isSystemInDarkTheme()) R.drawable.featuregraphic_dark else R.drawable.featuregraphic_light
    val pages = listOf(
        OnboardingModel(
            title = context.getString(R.string.welcome),
            description = context.getString(R.string.welcomeText),
            imageRes = fg),
        OnboardingModel(
            title = context.getString(R.string.usage),
            description = context.getString(R.string.usageText),
            imageRes = R.drawable.de_ic_usage),
        OnboardingModel(
            title = context.getString(R.string.disclaimer),
            description = context.getString(R.string.disclaimerText),
            imageRes = R.drawable.ic_disclaimer),
    )

    val pagerState = rememberPagerState(pageCount = { pages.size } )
    val coroutineScope = rememberCoroutineScope()

    Column(modifier = modifier.fillMaxSize().navigationBarsPadding().padding(16.dp)) {
        HorizontalPager(state = pagerState, modifier = Modifier.weight(1f).fillMaxWidth()) { page -> OnboardingItem(pages[page]) }
        Row(horizontalArrangement = Arrangement.SpaceBetween, verticalAlignment = Alignment.CenterVertically, modifier = Modifier.fillMaxWidth().padding(10.dp)
        ) {
            val textColor = if (pagerState.currentPage == 0) MaterialTheme.colorScheme.onPrimary else MaterialTheme.colorScheme.primary //Color(0xFFAAAAAA) else Color(0xFF333333)
            Text(
                context.getString(R.string.previous), style = TextStyle(color = textColor, fontSize = 14.sp, fontWeight = FontWeight.Normal),
                modifier = Modifier.clickable {
                    val previousPage = pagerState.currentPage - 1
                    coroutineScope.launch { pagerState.animateScrollToPage(previousPage) }
                }
            )

            Row(horizontalArrangement = Arrangement.Center, modifier = Modifier.weight(1f)) {
                repeat(pages.size) { index ->
                    val isSelected = pagerState.currentPage == index
                    Box(
                        modifier = Modifier
                            .padding(4.dp)
                            .width(if (isSelected) 18.dp else 8.dp)
                            .height(if (isSelected) 8.dp else 8.dp)
                            .border(width = 1.dp, color = Color(0xFF707784), shape = RoundedCornerShape(10.dp))
                            .background(color = if (isSelected) Color(0xFF3B6C64) else Color(0xFFFFFFFF), shape = CircleShape)
                    )
                }
            }
            val s  = if (pagerState.currentPage < pages.size -1) context.getString(R.string.next) else context.getString(R.string.done)
            Text(s, style = TextStyle(color = MaterialTheme.colorScheme.primary, fontSize = 14.sp, fontWeight = FontWeight.Normal),
                modifier = Modifier.clickable {
                    if (pagerState.currentPage < pages.size -1) {
                        val nextPage = pagerState.currentPage + 1
                        coroutineScope.launch { pagerState.animateScrollToPage(nextPage) }
                    }
                    else setValue(AppStart.NORMAL)
                }
            )

        }
    }
}
