package com.zell_mbc.publicartexplorer.oauth2

import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.util.Log
import androidx.core.net.toUri
import net.openid.appauth.*
import androidx.core.content.edit
import net.openid.appauth.CodeVerifierUtil // For generating PKCE parameters
import java.security.MessageDigest
import android.util.Base64
import android.widget.Toast
import com.zell_mbc.publicartexplorer.osmAppId
import com.zell_mbc.publicartexplorer.oAuthRedirectUri
import com.zell_mbc.publicartexplorer.osmAuthUrl
import com.zell_mbc.publicartexplorer.osmScope
import com.zell_mbc.publicartexplorer.sharedOkHttpClient
import com.zell_mbc.publicartexplorer.wikiAppId
import com.zell_mbc.publicartexplorer.wikiScope
import com.zell_mbc.publicartexplorer.wikimediaAuthEndpoint
import com.zell_mbc.publicartexplorer.wikimediaTokenEndpoint
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.security.NoSuchAlgorithmException
import java.security.SecureRandom
import net.openid.appauth.AuthorizationRequest
import okhttp3.FormBody
import okhttp3.Request
import org.json.JSONObject

const val AUTH_REQUEST_CODE_SUCCESS = 100
const val AUTH_REQUEST_CODE_FAILURE = 101

var serviceConfig: AuthorizationServiceConfiguration? = null

fun startWikiLogin(context: Context) {
    serviceConfig = AuthorizationServiceConfiguration(wikimediaAuthEndpoint.toUri(), wikimediaTokenEndpoint.toUri())
    oAuthLogin(context, clientId = wikiAppId, scope = wikiScope, serviceId = "Wiki")
}

fun startOsmLogin(context: Context) {
    serviceConfig = AuthorizationServiceConfiguration("$osmAuthUrl/authorize".toUri(), "$osmAuthUrl/token".toUri())
    oAuthLogin(context, clientId = osmAppId, scope = osmScope, serviceId = "OSM")
}

fun oAuthLogin(context: Context, clientId: String, scope: String, serviceId: String) {
    if (serviceConfig == null) return

    // 1. OAuth2 endpoints
    // Generate state (recommended for CSRF protection and can be used as a key)
    val state = generateRandomState()
    val codeVerifier = CodeVerifierUtil.generateRandomCodeVerifier() // This defines codeVerifier

    // 2. Generate PKCE parameters
    val codeChallenge: String
    try {
        val digest = MessageDigest.getInstance("SHA-256")
        val hash = digest.digest(codeVerifier.toByteArray(Charsets.US_ASCII))         // Make sure to use Charsets.US_ASCII for code_verifier as per PKCE spec
        codeChallenge = Base64.encodeToString(hash, Base64.URL_SAFE or Base64.NO_PADDING or Base64.NO_WRAP) // Correct usage of Base64 constants
    } catch (e: NoSuchAlgorithmException) {
        Log.e("PKCE_ERROR", "SHA-256 algorithm not found!", e)         // Handle error appropriately (e.g., show a message to the user)
        Toast.makeText(context, "Security error: Cannot initiate login.", Toast.LENGTH_LONG).show()
        return
    }

    // Store codeVerifier associated with state in SharedPreferences
    val preferences = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
    preferences.edit {
        putString("code_verifier_state", state + "|$serviceId") // Key with state
        putString("code_verifier_$state", codeVerifier) // Key with state
        Log.d("OAuthCallback", "Set code_verifier_$state")
        Log.d("OAuthCallback", "Set codeVerifier: $codeVerifier")
    }
    val additionalParams = HashMap<String, String>()
    additionalParams["prompt"] = "login"
    Log.d("OAuthCallback", "Scope: $scope")
    val authRequest = AuthorizationRequest.Builder(serviceConfig!!, clientId, ResponseTypeValues.CODE, oAuthRedirectUri.toUri())
        .setScope(scope)
        .setCodeVerifier(codeVerifier, codeChallenge, "S256")
        .build()

    val authService = AuthorizationService(context)

    val successIntent = PendingIntent.getActivity(context, AUTH_REQUEST_CODE_SUCCESS, Intent(context, OAuthCallbackActivity::class.java), PendingIntent.FLAG_MUTABLE)
    val failureIntent = PendingIntent.getActivity(context, AUTH_REQUEST_CODE_FAILURE, Intent(context, OAuthCallbackActivity::class.java), PendingIntent.FLAG_MUTABLE)

    // Launch OAuth flow
    authService.performAuthorizationRequest(authRequest, successIntent, failureIntent)
}

/**
 * Generates a cryptographically-secure random string to be used as the 'state'
 * parameter in an OAuth 2.0 authorization request. This helps prevent CSRF attacks.
 * The state value is also used here to key the code_verifier in SharedPreferences.
 */
private fun generateRandomState(): String {
    val random = SecureRandom() // Uses java.security.SecureRandom
    val bytes = ByteArray(16)   // 16 bytes = 128 bits, a good length for state
    random.nextBytes(bytes)
    // Uses android.util.Base64
    return Base64.encodeToString(
        bytes,
        Base64.URL_SAFE or Base64.NO_PADDING or Base64.NO_WRAP
    )
}

fun startWikiRenewal(context: Context) {
    val client = sharedOkHttpClient

    val body = FormBody.Builder()
        .add("grant_type", "refresh_token")
        .add("refresh_token",TokenManager.wikiToken.refreshToken ?: "")
        .add("client_id", wikiAppId)
        .build()

    val request = Request.Builder()
        .url("https://www.wikidata.org/w/rest.php/oauth2/access_token")
        .post(body)
        .build()

    CoroutineScope(Dispatchers.IO).launch {
        try {
            client.newCall(request).execute().use { response ->
                val body = response.body?.string()
                Log.e("WikiRefresh", "Code=${response.code}, Body=$body")
                if (!response.isSuccessful) return@use

                val json = JSONObject(response.body.string())

                val token = OAuthToken(
                    accessToken = json.getString("access_token"),
                    refreshToken = json.optString("refresh_token", ""),
                    expiresIn = json.getLong("expires_in"),
                    obtainedAt = System.currentTimeMillis()
                )

                // Switch back to Main for UI/state updates
                withContext(Dispatchers.Main) {
                    TokenManager.setWikiToken(context, token)
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
            Toast.makeText(context, "Unable to renew token: ${e.message}", Toast.LENGTH_LONG).show()
        }
    }
}