package com.zell_mbc.publicartexplorer.maps

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.location.Location
import android.os.Handler
import android.os.Looper
import android.widget.Toast
import androidx.core.content.ContextCompat
import com.google.android.gms.common.ConnectionResult
import com.google.android.gms.common.GoogleApiAvailability
import com.google.android.gms.location.LocationCallback
import com.google.android.gms.location.LocationRequest
import com.google.android.gms.location.LocationResult
import com.google.android.gms.location.LocationServices
import com.google.android.gms.location.Priority
import com.zell_mbc.publicartexplorer.R

fun getLocation(
    context: Context,
    callback: (Location?) -> Unit
) {
    val fineLocationGranted = ContextCompat.checkSelfPermission(
        context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED
    val coarseLocationGranted = ContextCompat.checkSelfPermission(
        context, Manifest.permission.ACCESS_COARSE_LOCATION) == PackageManager.PERMISSION_GRANTED

    if (!fineLocationGranted && !coarseLocationGranted) {
        callback(null)
        return
    }

    val googleApiAvailable = GoogleApiAvailability.getInstance()
        .isGooglePlayServicesAvailable(context) == ConnectionResult.SUCCESS

    if (!googleApiAvailable) {
        callback(null)
        return // No Google Play Services available, no fallback
    }

    val fusedLocationClient = LocationServices.getFusedLocationProviderClient(context)

    val locationRequest = LocationRequest.Builder(Priority.PRIORITY_HIGH_ACCURACY, 10000)
        .setMinUpdateIntervalMillis(5000)
        .setMaxUpdates(1)
        .build()

    var callbackCalled = false
    val handler = Handler(Looper.getMainLooper())
    val locationCallback = object : LocationCallback() {
        override fun onLocationResult(locationResult: LocationResult) {
            if (callbackCalled) return
            callbackCalled = true
            handler.removeCallbacksAndMessages(null)
            fusedLocationClient.removeLocationUpdates(this)

            val location = locationResult.locations.firstOrNull()
            callback(location)        }

        override fun onLocationAvailability(availability: com.google.android.gms.location.LocationAvailability) {
            if (!availability.isLocationAvailable) {
                if (callbackCalled) return
                callbackCalled = true
                handler.removeCallbacksAndMessages(null)
                fusedLocationClient.removeLocationUpdates(this)
                callback(null)
            }
        }
    }

    try {
        fusedLocationClient.requestLocationUpdates(locationRequest, locationCallback, Looper.getMainLooper())

        // Optional: add a timeout to stop listening after 20 seconds
        val handler = Handler(Looper.getMainLooper())
        handler.postDelayed({
            if (callbackCalled) return@postDelayed
            callbackCalled = true
            fusedLocationClient.removeLocationUpdates(locationCallback)
            // Toast must be shown on the main thread
            Handler(Looper.getMainLooper()).post { Toast.makeText(context, context.getString(R.string.unableToRetrieveLocation), Toast.LENGTH_LONG).show() }
            callback(null) // timeout fallback
        }, 20000)

    } catch (_: SecurityException) {
        callback(null)
    }
}
