package com.zell_mbc.publicartexplorer.wiki

import android.content.Context
import android.database.Cursor
import androidx.exifinterface.media.ExifInterface
import android.net.Uri
import android.util.Log
import androidx.core.content.edit
import com.zell_mbc.publicartexplorer.DebugLog
import com.zell_mbc.publicartexplorer.sharedOkHttpClient
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okhttp3.Request
import org.json.JSONObject
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale

object WikiUserObject {
    var id = 0L
    var name = ""
    var registration = ""
    var editcount = 0
    var groups: List<String> = emptyList()
    var rights: List<String> = emptyList()

    val canUpload: Boolean
        get() = rights.contains("upload")

    fun loadName(context: Context) {
        val prefs = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
        name = prefs.getString("wikiUsername", "") ?: ""
    }

    private const val BASE_URL =
        "https://commons.wikimedia.org/w/api.php?action=query&meta=userinfo&uiprop=registration|editcount|groups|rights&format=json"

    suspend fun getUserDetails(context: Context, token: String?) = withContext(Dispatchers.IO) {
        val request = Request.Builder()
            .url(BASE_URL)
            .header("Authorization", "Bearer $token")
            .build()

        return@withContext try {
            sharedOkHttpClient.newCall(request).execute().use { response ->
                if (!response.isSuccessful) {
                    Log.e("WikiApi", "HTTP error: ${response.code}")
                    return@use null
                }

                val body = response.body.string() //?: return@use null
                DebugLog.add(body)
                val root = JSONObject(body)
                val userInfo = root.getJSONObject("query").getJSONObject("userinfo")
                id = userInfo.getLong("id")
                name = userInfo.getString("name")
                registration = userInfo.optString("registration", "")
                editcount = userInfo.optInt("editcount", 0)
                groups = userInfo.optJSONArray("groups")?.let { arr ->
                    List(arr.length()) { idx -> arr.getString(idx) }
                } ?: emptyList()
                rights = userInfo.optJSONArray("rights")?.let { arr ->
                    List(arr.length()) { idx -> arr.getString(idx) }
                } ?: emptyList()

                val prefs = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
                prefs.edit { putString("wikiUsername", WikiUserObject.name) }
            }
        } catch (e: Exception) {
            Log.e("WikiApi", "Failed to fetch Wikimedia user details", e)
        }
    }
}

fun getImageCreationDateString(context: Context, uri: Uri): String {
    //dumpExif(context, uri)
    return try {
        context.contentResolver.openInputStream(uri)?.use { input ->
            val exif = ExifInterface(input)
            val dateStr = exif.getAttribute(ExifInterface.TAG_DATETIME_ORIGINAL)
                ?: exif.getAttribute(ExifInterface.TAG_DATETIME)

            dateStr?.let {
                val sdf = SimpleDateFormat("yyyy:MM:dd HH:mm:ss", Locale.US)
                val date = sdf.parse(it)
                if (date != null) {
                    SimpleDateFormat("yyyy-MM-dd", Locale.US).format(date)
                } else {
                    // fallback
                    SimpleDateFormat("yyyy-MM-dd", Locale.US).format(Date())
                }
            } ?: SimpleDateFormat("yyyy-MM-dd", Locale.US).format(Date())
        } ?: SimpleDateFormat("yyyy-MM-dd", Locale.US).format(Date())
    } catch (e: Exception) {
        e.printStackTrace()
        SimpleDateFormat("yyyy-MM-dd", Locale.US).format(Date())
    }
}

fun dumpExif(context: Context, uri: Uri) {
    try {
        context.contentResolver.openFileDescriptor(uri, "r")?.use { pfd ->
            val exif = ExifInterface(pfd.fileDescriptor)

            // Print all known EXIF tags
            val allTags = listOf(
                ExifInterface.TAG_DATETIME,
                ExifInterface.TAG_MAKE,
                ExifInterface.TAG_MODEL,
                ExifInterface.TAG_GPS_LATITUDE,
                ExifInterface.TAG_GPS_LATITUDE_REF,
                ExifInterface.TAG_GPS_LONGITUDE,
                ExifInterface.TAG_GPS_LONGITUDE_REF,
                ExifInterface.TAG_GPS_ALTITUDE,
                ExifInterface.TAG_GPS_ALTITUDE_REF,
                ExifInterface.TAG_GPS_TIMESTAMP,
                ExifInterface.TAG_GPS_DATESTAMP,
                ExifInterface.TAG_SOFTWARE
            )

            for (tag in allTags) {
                val value = exif.getAttribute(tag)
                Log.d("EXIF", "$tag: $value")
            }

            // Dump everything, including unknown/custom tags
            for (tag in exif.javaClass.fields) {
                if (tag.name.startsWith("TAG_")) {
                    val tagValue = exif.getAttribute(tag.get(null) as String)
                    if (tagValue != null) {
                        Log.d("EXIF_ALL", "${tag.name}: $tagValue")
                    }
                }
            }
        }
    } catch (e: Exception) {
        e.printStackTrace()
    }
}


private fun Cursor.getDoubleOrNull(index: Int): Double? {
    return if (isNull(index)) null else getDouble(index)
}


fun extractGpsFromImageUri(context: Context, uri: Uri): ImageLocation {
    return try {
        context.contentResolver.openInputStream(uri)?.use { input ->
            val exif = ExifInterface(input)

            // First try the standard helper
            exif.latLong?.let { return ImageLocation(it[0], it[1]) }

            // Fallback: parse rational strings manually
            val latStr = exif.getAttribute(ExifInterface.TAG_GPS_LATITUDE)
            val latRef = exif.getAttribute(ExifInterface.TAG_GPS_LATITUDE_REF)
            val lonStr = exif.getAttribute(ExifInterface.TAG_GPS_LONGITUDE)
            val lonRef = exif.getAttribute(ExifInterface.TAG_GPS_LONGITUDE_REF)

            if (!latStr.isNullOrEmpty() && !lonStr.isNullOrEmpty() && !latRef.isNullOrEmpty() && !lonRef.isNullOrEmpty()) {
                val lat = parseExifRational(latStr, latRef)
                val lon = parseExifRational(lonStr, lonRef)
                ImageLocation(lat, lon)
            } else {
                ImageLocation(null, null)
            }
        } ?: ImageLocation(null, null)
    } catch (e: Exception) {
        e.printStackTrace()
        ImageLocation(null, null)
    }
}

// Convert EXIF rational string to decimal degrees
private fun parseExifRational(coord: String, ref: String): Double {
    val parts = coord.split(",")
    if (parts.size != 3) return 0.0

    fun rationalToDouble(r: String): Double {
        val nums = r.split("/")
        return if (nums.size == 2 && nums[1].toDoubleOrNull() != 0.0) {
            nums[0].toDouble() / nums[1].toDouble()
        } else nums[0].toDoubleOrNull() ?: 0.0
    }

    val degrees = rationalToDouble(parts[0])
    val minutes = rationalToDouble(parts[1])
    val seconds = rationalToDouble(parts[2])
    var decimal = degrees + minutes / 60.0 + seconds / 3600.0
    if (ref.uppercase() == "S" || ref.uppercase() == "W") decimal = -decimal
    return decimal
}


data class ImageLocation(
    val latitude: Double?,
    val longitude: Double?
)


fun fetchCategories(location: ImageLocation): List<String> {
    if (location.latitude == null || location.longitude == null) return emptyList()

    val lat = location.latitude
    val lon = location.longitude
    // Todo: consider makeing radius a variable
    val url = "https://commons.wikimedia.org/w/api.php" +
            "?action=query&list=geosearch&gscoord=$lat|$lon&gsradius=2000&gslimit=20&format=json"

    return try {
        val client = sharedOkHttpClient
        val request = Request.Builder().url(url).build()
        val response = client.newCall(request).execute()

        val body = response.body.string()
        val json = JSONObject(body)
        val geosearch = json.optJSONObject("query")?.optJSONArray("geosearch") ?: return emptyList()

        val results = mutableListOf<String>()

        for (i in 0 until geosearch.length()) {
            val obj = geosearch.getJSONObject(i)

            // 1️⃣ Add title for later "Public art in <title>" category
            val title = obj.optString("title")
            if (title.isNotEmpty()) results.add("Public art in $title")

            // 2️⃣ Add categories verbatim
            val categories = obj.optJSONArray("categories") // if the API provides them
            if (categories != null) {
                for (j in 0 until categories.length()) {
                    val cat = categories.optString(j)
                    if (cat.isNotEmpty()) results.add(cat)
                }
            }
        }
        for (item in results) DebugLog.add(item)
        return results

    } catch (e: Exception) {
        e.printStackTrace()
        emptyList()
    }
}
