package com.zell_mbc.publicartexplorer.screens

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.AccessAlarm
import androidx.compose.material.icons.filled.AccessTime
import androidx.compose.material.icons.filled.AccountCircle
import androidx.compose.material.icons.filled.CloudUpload
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material.icons.filled.Person
import androidx.compose.material.icons.filled.PersonOff
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.oauth2.TokenManager
import com.zell_mbc.publicartexplorer.wiki.WikiUserObject
import java.text.DateFormat
import java.util.Date
import java.util.Locale

data class WikiUser(
    val id: Long,
    val name: String,
    val registration: String?,
    val editcount: Int,
    val groups: List<String>,
    val rights: List<String>
) {
    val canUpload: Boolean
        get() = rights.contains("upload")
}

@Composable
fun WikiSection(modifier: Modifier = Modifier) {
    val context = LocalContext.current

    var accountLoaded by remember { mutableStateOf(false) }
    var error by remember { mutableStateOf<String?>(null) }
    var message by remember { mutableIntStateOf(R.string.notLoggedIn) }

    if (!TokenManager.wikiToken.isExpired()) {
        LaunchedEffect(TokenManager.wikiToken.accessToken) {
            try {
                WikiUserObject.getUserDetails(context, TokenManager.wikiToken.accessToken)
                if (WikiUserObject.name.isNotEmpty()) {
                    accountLoaded = true
                } else error = "Failed to load Wikidata user info"
            } catch (e: Exception) {
                error = e.message
            }
            message = R.string.loadingUserInfo
        }
    }
    else message = R.string.notLoggedIn

    Surface(
        modifier = modifier.fillMaxWidth(),
        tonalElevation = 6.dp,
        shape = MaterialTheme.shapes.medium
    ) {
        Column(
            modifier = Modifier.padding(20.dp),
            verticalArrangement = Arrangement.spacedBy(16.dp)
        ) {
            Text(
                "Wikimedia Commons",
                style = MaterialTheme.typography.titleMedium,
                color = MaterialTheme.colorScheme.primary
            )
            when {
                accountLoaded -> {
                    ItemRow(icon = Icons.Default.AccountCircle, label = stringResource(R.string.userId), value = WikiUserObject.id.toString())
                    ItemRow(icon = Icons.Default.Person, label = stringResource(R.string.userName), value = WikiUserObject.name)
                    //ItemRow(icon = Icons.Default.DateRange, label = stringResource(R.string.accountCreated), value = formattedDate)
                    ItemRow(icon = Icons.Default.Edit, label = stringResource(R.string.changeSets), value = WikiUserObject.editcount.toString())
                    ItemRow(icon = Icons.Default.CloudUpload, label = stringResource(R.string.wikiUploadRights), value = if (WikiUserObject.canUpload) stringResource(R.string.allowed) else stringResource(R.string.notAllowed))
                    when {
                        TokenManager.wikiToken.accessToken.isNullOrEmpty() -> ItemRow(icon = Icons.Default.PersonOff, label = stringResource(R.string.notLoggedIn), value = "")
                        TokenManager.wikiToken.isExpired() -> ItemRow(icon = Icons.Default.AccessAlarm, label = stringResource(R.string.expired), value = "")
                        !TokenManager.wikiToken.isExpired() -> ItemRow(icon = Icons.Default.AccessTime, label = stringResource(R.string.tokenExpires), value = formatMillis(TokenManager.wikiToken.expiresIn))
                    }
                }
                error != null -> Text("Error: $error", color = MaterialTheme.colorScheme.error)
                else -> Text(stringResource(message), style = MaterialTheme.typography.bodyMedium, color = MaterialTheme.colorScheme.onSurfaceVariant)
            }
        }
    }
}

fun formatMillis(millis: Long): String {
    val date = Date(millis)
    val formatter = DateFormat.getDateTimeInstance(
        DateFormat.MEDIUM,  // date style (Medium = "Sep 27, 2025")
        DateFormat.SHORT,   // time style (Short = "5:30 PM")
        Locale.getDefault() // respect device locale
    )
    return formatter.format(date)
}