package com.zell_mbc.publicartexplorer.screens

import PreferencesManager
import android.content.Context
import android.widget.Toast
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.runtime.*
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.hapticfeedback.HapticFeedbackType
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalHapticFeedback
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.core.content.edit
import com.zell_mbc.publicartexplorer.*
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.billing.BillingUnavailableDialog
import com.zell_mbc.publicartexplorer.billing.HandleSubscription
import com.zell_mbc.publicartexplorer.billing.ThankYouForPurchaseDialog
import com.zell_mbc.publicartexplorer.data.ViewModel
import com.zell_mbc.publicartexplorer.maps.getAvailableMapboxStyles
import com.zell_mbc.publicartexplorer.oauth2.TokenManager
import com.zell_mbc.publicartexplorer.oauth2.startOsmLogin
import com.zell_mbc.publicartexplorer.oauth2.startWikiLogin
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun SettingsScreen(viewModel: ViewModel) {
    val context = LocalContext.current
    val preferences = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
    val prefs = PreferencesManager(context) // Datastore Preferences
    val scrollState = rememberScrollState()
    val hapticFeedback = LocalHapticFeedback.current

    Column(modifier = Modifier
        .fillMaxSize()
        .verticalScroll(scrollState)
        .padding(16.dp), verticalArrangement = Arrangement.spacedBy(24.dp)) {
        Text(
            "Settings",
            style = MaterialTheme.typography.headlineMedium,
            color = MaterialTheme.colorScheme.primary
        )

        //ThemeSettingsUI()

        val darkModeOptionStrings = listOf(
            stringResource(R.string.light),
            stringResource(R.string.dark),
            stringResource(R.string.followSystem)
        )
        ListPreference(
            names = darkModeOptionStrings.map { it },
            selectedIndex = viewModel.darkMode,
            label = stringResource(R.string.darkMode),
            onItemSelected = { index ->
                viewModel.darkMode = index
                preferences.edit { putInt(DARK_MODE_KEY, index) }
            }
        )


        // Radius Slider Setting
        SettingSlider(
            label = stringResource(R.string.radius),
            value = viewModel.radius.toFloat(),
            unit = "m",
            valueRange = 100f..10000f,
            onValueChange = {
                viewModel.radius = it.toInt()
                preferences.edit { putInt(RADIUS_KEY, viewModel.radius) }
            }
        )

        // Show Museum Switch
        SettingSwitch(
            title = stringResource(R.string.showMuseum),
            summary = stringResource(R.string.showMuseumSummary),
            checked = viewModel.showMuseum,
            onCheckedChange = {
                if (viewModel.enableHapticFeedback) hapticFeedback.performHapticFeedback(
                    HapticFeedbackType.Confirm
                )
                viewModel.showMuseum = it
                preferences.edit { putBoolean(SHOW_MUSEUM_KEY, viewModel.showMuseum) }
            }
        )

        val flavour = BuildConfig.FLAVOR

        var poiServerIndex = poiServers.indexOfFirst { it.name == viewModel.selectedPoiServer }
        if (poiServerIndex < 0) poiServerIndex = 0
        if (viewModel.expertMode && BuildConfig.FLAVOR == "foss") // Disable selector for Play until Maptoolkit situation is clear
            ListPreference(
                names = poiServers.map { it.name },
                selectedIndex = poiServerIndex,
                label = stringResource(R.string.selectPoiServer),
                onItemSelected = { index ->
                    poiServerIndex = index
                    viewModel.selectedPoiServer = poiServers[index].name
                    preferences.edit { putString(POI_SERVER_KEY, viewModel.selectedPoiServer) }
                }
            )

        if (viewModel.expertMode) {
            ListPreference(
                names = tileServers.map { it.name },
                selectedIndex = viewModel.selectedTileServer,
                label = stringResource(R.string.selectTileServer),
                onItemSelected = { index ->
                    viewModel.selectedTileServer = index
                    preferences.edit { putInt(TILE_SERVER_KEY, viewModel.selectedTileServer) }
                }
            )
            // Show API field only if FOSS build. Play builds won't need it as they will use my keys
            if (viewModel.selectedTileServer != OPENFREEMAP_TILE_SERVER && (BuildConfig.FLAVOUR == "foss")) {
                val apiPreference = remember(viewModel.selectedTileServer) {
                    when (viewModel.selectedTileServer) {
                        MAPBOX_TILE_SERVER -> MAPBOX_API_KEY
                        else -> MAPTOOLKIT_API_KEY
                    }
                }

                // Keep API key as state so it survives recompositions
                var apiKey by remember(apiPreference) { mutableStateOf(preferences.getString(apiPreference, "").orEmpty()) }
                ApiKeyInputField(
                    label = "API Key",
                    apiKey = apiKey,
                    onApiKeyChange = {
                        apiKey = it
                        preferences.edit { putString(apiPreference, apiKey) } }
                )
            }

        }
        // Map style selector
        if (viewModel.selectedTileServer < 0) viewModel.selectedTileServer = 0
        val styleList = when (viewModel.selectedTileServer) {
            MAPBOX_TILE_SERVER -> getAvailableMapboxStyles(context)
            MAPTOOLKIT_TILE_SERVER -> mapToolKitStyles
            else -> openFreeMapStyles
        }

        if (viewModel.selectedMapStyleIndex >= styleList.size) viewModel.selectedMapStyleIndex = 0
        ListPreference(
            names = styleList.map { it },
            selectedIndex = viewModel.selectedMapStyleIndex,
            label = stringResource(R.string.selectMapStyle),
            onItemSelected = { index ->
                viewModel.selectedMapStyleIndex = index
                preferences.edit { putInt(MAP_STYLE_KEY, index) }
            }
        )

        // Expert Mode Switch (conditionally shown)
        if (flavour == "foss" || viewModel.subscriptionActive) {
            SettingSwitch(
                title = stringResource(R.string.expertMode),
                summary = stringResource(R.string.expertModeSummary),
                checked = viewModel.expertMode,
                onCheckedChange = {
                    if (viewModel.enableHapticFeedback) hapticFeedback.performHapticFeedback(
                        HapticFeedbackType.Confirm
                    )
                    viewModel.expertMode = it
                    preferences.edit { putBoolean(EXPERT_MODE_KEY, viewModel.expertMode) }
                }
            )
        }
        else {
            HandleSubscription(viewModel)
        }

        if (viewModel.expertMode) {
            SettingSwitch(
                title = stringResource(R.string.sandboxMode),
                summary = stringResource(R.string.sandboxModeSummary),
                checked = viewModel.sandboxMode,
                onCheckedChange = {
                    if (viewModel.enableHapticFeedback) hapticFeedback.performHapticFeedback(
                        HapticFeedbackType.Confirm
                    )
                    viewModel.sandboxMode = it
                    //preferences.edit { putBoolean(SANDBOX_MODE_KEY, viewModel.expertMode) }
                }
            )

            var osmButtonText by remember { mutableStateOf (if (TokenManager.osmToken.accessToken.isNullOrEmpty()) R.string.login else R.string.logoff) }
            var wikiButtonText by remember { mutableStateOf (if (!TokenManager.wikiToken.isExpired()) R.string.logoff else R.string.login)}

            Column {
            HorizontalDivider(thickness = 1.dp, color = MaterialTheme.colorScheme.primary)
            Spacer(Modifier.height(16.dp))

            SettingButton(
                title = "OpenStreetMap",
                summary = stringResource(R.string.osmLogonDescription),
                buttonText = stringResource(if (TokenManager.osmToken.accessToken.isNullOrEmpty()) R.string.login else R.string.logoff),
                onClick = {
                    if (TokenManager.osmToken.accessToken.isNullOrEmpty()) startOsmLogin(context)
                    else {
                        TokenManager.deleteOsmToken(context)
                        prefs.setString(OSM_ACCESS_TOKEN_KEY, "", CoroutineScope(Dispatchers.IO))
                        Toast.makeText(context, context.getString(R.string.tokenDeleted), Toast.LENGTH_SHORT).show()
                        osmButtonText = R.string.login
                    }
                }
            )
            SettingButton(
                title = "Wikidata",
                summary = stringResource(R.string.wikiLogonDescription),
                buttonText = stringResource(if (!TokenManager.wikiToken.isExpired()) R.string.logoff else R.string.login),
                onClick = {
                    if (!TokenManager.wikiToken.isExpired())  {
                            TokenManager.deleteWikiToken(context) //wikiToken.accessToken = ""
                            TokenManager.saveWikiToken(context)
                            Toast.makeText(context, context.getString(R.string.tokenDeleted), Toast.LENGTH_SHORT).show()
                            wikiButtonText  = R.string.login
                        }
                    else startWikiLogin(context) // Start OAuth login flow
                }
            )
        }
        }
        if (viewModel.showThankYouDialogState) ThankYouForPurchaseDialog(onDismiss = { viewModel.showThankYouDialogState = false })
        if (viewModel.showBillingUnavailableDialogState) BillingUnavailableDialog(onDismiss = { viewModel.showBillingUnavailableDialogState = false })
    }
}
