package com.zell_mbc.publicartexplorer.screens

import android.content.Context
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Icon
import androidx.compose.material3.NavigationBar
import androidx.compose.material3.NavigationBarItem
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.compositionLocalOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import com.zell_mbc.publicartexplorer.DEBUG_MODE_KEY
import com.zell_mbc.publicartexplorer.DebugOverlay
import com.zell_mbc.publicartexplorer.Screen
import com.zell_mbc.publicartexplorer.bottomNavItems
import com.zell_mbc.publicartexplorer.data.ViewModel

val LocalSnackbarHostState = compositionLocalOf<SnackbarHostState> {
    error("SnackbarHostState not provided")
}

@Composable
fun MainScaffold(viewModel: ViewModel) {
    val context = LocalContext.current
    val snackbarHostState = remember { SnackbarHostState() }

    val navController = rememberNavController()
    // Provide the LocalSnackbarHostState so children can read it
    CompositionLocalProvider(LocalSnackbarHostState provides snackbarHostState) {
        Scaffold(
            snackbarHost = { SnackbarHost(snackbarHostState) },
            bottomBar = {
                NavigationBar {
                    val navBackStackEntry by navController.currentBackStackEntryAsState()
                    val currentRoute = navBackStackEntry?.destination?.route
                    bottomNavItems.forEach { screen ->
                        NavigationBarItem(
                            icon = {
                                Icon(
                                    painterResource(id = screen.icon),
                                    contentDescription = null
                                )
                            },
                            label = { Text(context.getString(screen.label)) },
                            selected = currentRoute == screen.route,
                            onClick = {
                                if (currentRoute != screen.route) {
                                    navController.navigate(screen.route) {
                                        popUpTo(navController.graph.startDestinationId)
                                        launchSingleTop = true
                                    }
                                }
                            }
                        )
                    }
                }
            }
        ) { innerPadding ->
            NavHost(
                navController,
                startDestination = Screen.Map.route,
                Modifier.padding(innerPadding)
            ) {
                composable(Screen.Map.route) {
                    val preferences = context.getSharedPreferences(
                        "com.zell_mbc.publicartexplorer._preferences",
                        Context.MODE_PRIVATE
                    )
                    viewModel.debugMode = preferences.getBoolean(DEBUG_MODE_KEY, false)

                    if (viewModel.debugMode)
                        DebugOverlay(viewModel) {
                            MapScreen(snackbarHostState, viewModel)
                        }
                    else MapScreen(snackbarHostState, viewModel)
                }
                composable(Screen.Settings.route) { SettingsScreen(viewModel) }
                composable(Screen.Info.route) { InfoScreen(viewModel) }
            }
        }
    }
}
