package com.zell_mbc.publicartexplorer.screens

import android.content.Intent
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Email
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLinkStyles
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.withLink
import androidx.compose.ui.unit.dp
import com.zell_mbc.publicartexplorer.ProjectDetails
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.currentLocale
import androidx.core.net.toUri
import com.zell_mbc.publicartexplorer.ui.theme.link

@Composable
fun ContactSection() {
    val context = LocalContext.current
    val iconButtonModifier = Modifier.Companion.size(30.dp) // smaller size than default 48dp

    var url = ProjectDetails.AUTHOR_URL
    val currentLanguage = currentLocale(context)
    url += when (currentLanguage) {
        "de" -> "/de"
        else -> ""
    }

    Surface(modifier = Modifier.fillMaxWidth(), tonalElevation = 6.dp, shape = MaterialTheme.shapes.medium) {
        Column(modifier = Modifier.padding(16.dp), verticalArrangement = Arrangement.spacedBy(12.dp)) {
            Text(text = stringResource(R.string.contact), style = MaterialTheme.typography.titleMedium, color = MaterialTheme.colorScheme.primary)
            Text(buildAnnotatedString {
                append(stringResource(R.string.author) + ": ")
                withLink(LinkAnnotation.Url(url = url, TextLinkStyles(style = SpanStyle(color = MaterialTheme.colorScheme.link))
                )) { append(ProjectDetails.AUTHOR) }
            }, style = MaterialTheme.typography.bodyMedium, textDecoration = TextDecoration.Underline)

            Column(verticalArrangement = Arrangement.spacedBy(0.dp), modifier = Modifier.Companion.padding(0.dp)) {
                // Email Row
                Row(verticalAlignment = Alignment.Companion.CenterVertically, modifier = Modifier.Companion.height(IntrinsicSize.Min)) {
                    IconButton(
                        onClick = {
                            val intent = Intent(
                                Intent.ACTION_VIEW,
                                ("mailto:" + ProjectDetails.EMAIL_ADDRESS).toUri()
                            )
                            context.startActivity(intent)
                        },
                        modifier = iconButtonModifier
                    ) {
                        Icon(
                            imageVector = Icons.Outlined.Email,
                            contentDescription = stringResource(R.string.emailDescription),
                            modifier = Modifier.Companion.size(24.dp) // icon size inside button
                        )
                    }
                    Text(" " + stringResource(R.string.emailDescription), style = MaterialTheme.typography.bodyMedium, color = MaterialTheme.colorScheme.primary, modifier = Modifier.Companion.padding(vertical = 0.dp)) // no extra vertical padding)
                }

                // Fediverse Row
                Row(verticalAlignment = Alignment.Companion.CenterVertically, modifier = Modifier.Companion.height(IntrinsicSize.Min)) {
                    IconButton(
                        onClick = {
                            val intent =
                                Intent(Intent.ACTION_VIEW, ProjectDetails.FEDIVERSE_URL.toUri())
                            context.startActivity(intent)
                        },
                        modifier = iconButtonModifier
                    ) {
                        Icon(
                            painterResource(R.drawable.fediverse_icon),
                            contentDescription = stringResource(R.string.fediverseDescription),
                            modifier = Modifier.Companion.size(24.dp)
                        )
                    }
                    Text(
                        " " + stringResource(R.string.fediverseDescription),
                        style = MaterialTheme.typography.bodyMedium,
                        color = MaterialTheme.colorScheme.primary,
                        modifier = Modifier.Companion.padding(vertical = 0.dp)
                    )
                }

                // Website Row
                Row(
                    verticalAlignment = Alignment.Companion.CenterVertically,
                    modifier = Modifier.Companion.height(IntrinsicSize.Min)
                ) {
                    IconButton(
                        onClick = {
                            val intent =
                                Intent(Intent.ACTION_VIEW, ProjectDetails.WEBSITE_URL.toUri())
                            context.startActivity(intent)
                        },
                        modifier = iconButtonModifier
                    ) {
                        Icon(
                            painter = painterResource(R.drawable.outline_web_24),
                            contentDescription = stringResource(R.string.websiteDescription),
                            modifier = Modifier.Companion.size(24.dp)
                        )
                    }
                    Text(
                        " " + stringResource(R.string.websiteDescription),
                        style = MaterialTheme.typography.bodyMedium,
                        color = MaterialTheme.colorScheme.primary,
                        modifier = Modifier.Companion.padding(vertical = 0.dp)
                    )
                }
            }
        }
    }
}