package com.zell_mbc.publicartexplorer.onboarding

import android.content.Context
import android.os.Build
import android.util.Log
import android.content.pm.PackageManager
import androidx.core.content.edit

enum class AppStart {
    FIRST_TIME_APP, FIRST_TIME_VERSION, NORMAL
}
const val NEVER_RUN = -1
const val BUILD_KEY = "app_build"

/**
 * Finds out started for the first time (ever or in the current version)
 *
 * Note: This method is **not idempotent** only the first call will
 * determine the proper result. Any subsequent calls will only return
 * [AppStart.NORMAL] until the app is started again. So you might want
 * to consider caching the result!
 *
 * @return the type of app start
 */

fun checkAppStart(context: Context): AppStart {
    val preferences = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)

    val lastBuild = preferences.getInt(BUILD_KEY, NEVER_RUN)

    val packageInfo = if (Build.VERSION.SDK_INT >= 33) context.packageManager.getPackageInfo(context.packageName, PackageManager.PackageInfoFlags.of(0))
    else {
        @Suppress("DEPRECATION")
        context.packageManager.getPackageInfo(context.packageName, 0)
    }
    val currentBuild = if (Build.VERSION.SDK_INT >= 28) packageInfo.longVersionCode.toInt()
    else {
        @Suppress("DEPRECATION")
        packageInfo.versionCode
    }

    preferences.edit { putInt(BUILD_KEY, currentBuild) }

    return when {
        lastBuild == NEVER_RUN -> AppStart.FIRST_TIME_APP
        lastBuild < currentBuild -> AppStart.FIRST_TIME_VERSION
        lastBuild > currentBuild -> {
            Log.w("Warning", "Current version code ($currentBuild) is less then the one recognized on last startup ($lastBuild). Defensively assuming normal app start.")
            AppStart.NORMAL
        }
        else -> {
            AppStart.NORMAL
        }
    }
}
