package com.zell_mbc.publicartexplorer.oauth2

import android.content.Context
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.content.edit

data class OAuthToken(
    var accessToken: String?,
    var refreshToken: String?,
    var expiresIn: Long,
    var obtainedAt: Long = System.currentTimeMillis()
) {
    fun isExpired(): Boolean {
        val now = System.currentTimeMillis()
        return now > expiresIn //obtainedAt + (expiresIn - 60) * 1000 // refresh 1 min early
    }
}

object TokenManager {
    const val osmPrefix = "osm"
    const val wikiPrefix = "wiki"

    var wikiToken by mutableStateOf<OAuthToken>(OAuthToken("", "", 0L, 0L))
    var osmToken by mutableStateOf<OAuthToken>(OAuthToken("", "", 0L, 0L))

    fun setWikiToken(context: Context, token: OAuthToken) {
        wikiToken = token
        saveToken(context, wikiToken, wikiPrefix)
    }

    fun saveWikiToken(context: Context) { saveToken(context, wikiToken, wikiPrefix) }
    fun loadWikiToken(context: Context) { wikiToken = loadToken(context,  wikiPrefix) }

    fun deleteWikiToken(context: Context) {
        wikiToken.accessToken = ""
        wikiToken.refreshToken = ""
        wikiToken.expiresIn = 0L
        wikiToken.obtainedAt = 0L
        saveWikiToken(context)
    }

    fun setOsmToken(context: Context, token: OAuthToken) {
        osmToken = token
        saveToken(context, osmToken, osmPrefix)
    }
    fun saveOsmToken(context: Context) { saveToken(context, osmToken, osmPrefix) }
    fun loadOsmToken(context: Context) { osmToken = loadToken(context,  osmPrefix) }

    fun deleteOsmToken(context: Context) {
        osmToken.accessToken = ""
        osmToken.refreshToken = ""
        osmToken.expiresIn = 0L
        osmToken.obtainedAt = 0L
        saveOsmToken(context)
    }

    private fun saveToken(context: Context, token: OAuthToken, keyPrefix: String) {
        val prefs = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
        prefs.edit {
            putString("${keyPrefix}_access_token", token.accessToken)
                .putString("${keyPrefix}_refresh_token", token.refreshToken)
                .putLong("${keyPrefix}_expires_in", token.expiresIn)
                .putLong("${keyPrefix}_obtained_at", token.obtainedAt)
        }
    }

    private fun loadToken(context: Context, keyPrefix: String): OAuthToken {
        val prefs = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
        return OAuthToken(
            accessToken = prefs.getString("${keyPrefix}_access_token", "").orEmpty(),
            refreshToken = prefs.getString("${keyPrefix}_refresh_token", "").orEmpty(),
            expiresIn = prefs.getLong("${keyPrefix}_expires_in", 0L ),
            obtainedAt = prefs.getLong("${keyPrefix}_obtained_at", 0L))
    }
}
