package com.zell_mbc.publicartexplorer.oauth2

import PreferencesManager
import android.content.Intent
import android.os.Bundle
import android.util.Log
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.appcompat.app.AlertDialog
import androidx.core.content.edit
import androidx.lifecycle.lifecycleScope
import kotlinx.coroutines.launch
import net.openid.appauth.AuthorizationService
import net.openid.appauth.TokenRequest
import androidx.core.net.toUri
import com.zell_mbc.publicartexplorer.OSM_ACCESS_TOKEN_KEY
import com.zell_mbc.publicartexplorer.DebugLog
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.osmAppId
import com.zell_mbc.publicartexplorer.oAuthRedirectUri
import com.zell_mbc.publicartexplorer.wiki.WikiUserObject
import com.zell_mbc.publicartexplorer.wikiAppId

class OAuthCallbackActivity : ComponentActivity() {
    private lateinit var authService: AuthorizationService

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        authService = AuthorizationService(this)
        Log.d("OAuthCallback", "OnCreate")

        handleAuthIntent(intent)
    }

    override fun onNewIntent(intent: Intent) {
        super.onNewIntent(intent)
        Log.d("OAuthCallback", "OnNewIntent")
        handleAuthIntent(intent)
    }

    private fun handleAuthIntent(intent: Intent) {
        val uri = intent.data
        Log.d("OAuthCallback", "Redirect URI: $uri")

        if (uri == null) {
            // Code comes by here when the browser window is closed. Hence we shouldn't alert the user
            finish()
            return
        }

        val code = uri.getQueryParameter("code")
        val error = uri.getQueryParameter("error")

        when {
            code != null -> {
                //DebugLog.add("OAuthCallback, Authorization code: $code")
                //Toast.makeText(this, "Received OAuth code", Toast.LENGTH_SHORT).show()
                exchangeTokenManually(code)
            }
            error != null -> {
                DebugLog.add("OAuthCallback, OAuth error: $error")
                Toast.makeText(this, "OAuth error: $error", Toast.LENGTH_SHORT).show()
                finish()
            }
            else -> {
                DebugLog.add("OAuthCallback, No code or error in redirect")
                finish()
            }
        }
    }

    private fun exchangeTokenManually(code: String) {
        if (serviceConfig == null) return

        // Retrieve codeVerifier associated with state in SharedPreferences
        val preferences = getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", MODE_PRIVATE)
        val state         = preferences.getString("code_verifier_state", "")
        val tag = state?.split("|")?.getOrNull(0) ?: "" // Remove serviceId
        val codeVerifier  = preferences.getString("code_verifier_$tag", "")

        val serviceId = state?.split("|")?.getOrNull(1) ?: "unknown"
        val clientId = when (serviceId) {
            "OSM" -> osmAppId
            "Wiki" -> wikiAppId
            else -> "unknown"
        }

        try {
            Log.d("OAuthCallback", "Get code_verifier_$state")
            Log.d("OAuthCallback", "Read codeVerifier: $codeVerifier")
            val tokenRequest = TokenRequest.Builder(serviceConfig!!, clientId)
                .setAuthorizationCode(code)
                .setRedirectUri(oAuthRedirectUri.toUri())
                .setCodeVerifier(codeVerifier)
                .build()

            authService.performTokenRequest(tokenRequest) { tokenResponse, exception ->
                runOnUiThread {
                    val prefs = PreferencesManager(this)
                    if (tokenResponse != null) {
                        lifecycleScope.launch {
                            when (serviceId) {
                                "OSM" -> TokenManager.setOsmToken(this@OAuthCallbackActivity, OAuthToken(
                                    accessToken = tokenResponse.accessToken, refreshToken = tokenResponse.refreshToken, expiresIn = tokenResponse.accessTokenExpirationTime ?: 0L))
                                "Wiki" -> {
                                    TokenManager.setWikiToken(this@OAuthCallbackActivity, OAuthToken(
                                        accessToken = tokenResponse.accessToken, refreshToken = tokenResponse.refreshToken, expiresIn = tokenResponse.accessTokenExpirationTime ?: 0L))
                                    WikiUserObject.getUserDetails(this@OAuthCallbackActivity, TokenManager.wikiToken.accessToken)
                                }
                            }
                        }
                        Toast.makeText(this, getString(R.string.loginSuccessful), Toast.LENGTH_SHORT).show()
                        /*android.app.AlertDialog.Builder(this) // Using the regular AlertDialog crashes the app bc. the activity isn't derived from AppCompat
                            .setTitle("Login Successful")
                            .setMessage(getString(R.string.loginSuccessful))
                            .setPositiveButton(getString(R.string.ok)) { dialog, _ -> finish() }
                            .show()*/

                    } else if (exception != null) {
                        var displayError = getString(R.string.tokenExchangeFailed)
                        DebugLog.add("OAuthCallback, $displayError AppAuth Exception Code: ${exception.code}")
                        DebugLog.add("OAuthCallback, $displayError AppAuth Exception Type: ${exception.type}")
                        DebugLog.add("OAuthCallback, $displayError AppAuth Error: ${exception.error}")
                        DebugLog.add("OAuthCallback, $displayError AppAuth Error Description: ${exception.errorDescription}")
                        DebugLog.add("OAuthCallback, $displayError AppAuth Error URI: ${exception.errorUri}")
                        DebugLog.add("OAuthCallback, $displayError Full Exception: $exception")

                        if (exception.errorDescription != null && exception.errorDescription!!.isNotEmpty()) {
                            displayError += ": ${exception.errorDescription}"
                        } else if (exception.error != null && exception.error!!.isNotEmpty()) {
                            displayError += ": ${exception.error}"
                        } else if (exception.message != null && exception.message!!.isNotEmpty()) {
                            // Fallback to general message if specific OAuth fields are null
                            displayError += ": ${exception.message}"
                        }
                        lifecycleScope.launch {
                            prefs.setString(OSM_ACCESS_TOKEN_KEY, "")
                        } // Invalidate token
                        Toast.makeText(this, displayError, Toast.LENGTH_LONG).show()
                    }
                    //authService.dispose()
                    preferences.edit {
                        remove("code_verifier_$state")
                        remove("code_verifier_state")
                    }
                    finish()
                }
            }
        } catch (e:Exception) {
            DebugLog.add(e.message.toString())
            Toast.makeText(this, getString(R.string.error) + ": " + e.message, Toast.LENGTH_LONG).show()
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        authService.dispose() // <- this unbinds the CustomTabs service
    }
}
