package com.zell_mbc.publicartexplorer.maps

import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.PorterDuff
import android.graphics.PorterDuffColorFilter
import android.graphics.Rect
import android.location.Location
import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.OpenInNew
import androidx.compose.material.icons.filled.Info
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.Divider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.core.content.ContextCompat
import androidx.core.graphics.createBitmap
import com.zell_mbc.publicartexplorer.DEFAULT_MAPTOOLKIT_STYLE
import com.zell_mbc.publicartexplorer.DEFAULT_OPENFREEMAP_STYLE
import com.zell_mbc.publicartexplorer.MAPBOX_TILE_SERVER
import com.zell_mbc.publicartexplorer.MAPTOOLKIT_TILE_SERVER
import com.zell_mbc.publicartexplorer.OPENFREEMAP_TILE_SERVER
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.mapToolKitStyles
import com.zell_mbc.publicartexplorer.mapboxStyles
import com.zell_mbc.publicartexplorer.openFreeMapStyles
import com.zell_mbc.publicartexplorer.tileServers
import org.maplibre.android.geometry.LatLng
import kotlin.math.asin
import kotlin.math.cos
import kotlin.math.pow
import kotlin.math.sin
import kotlin.math.sqrt
import androidx.core.net.toUri
import com.zell_mbc.publicartexplorer.maptoolkitName

fun createCircledIcon(context: Context, iconResId: Int, backgroundColor: Int, foregroundColor: Int): Bitmap { //Drawable {
    fun dpToPx(context: Context, dp: Int): Int {
        val metrics = context.resources.displayMetrics
        return (dp * metrics.density).toInt()
    }

    val sizePx = dpToPx(context, 36)

    // Load your icon as a Bitmap
    val iconDrawable = ContextCompat.getDrawable(context, iconResId)!!
    val iconBitmap = createBitmap(sizePx, sizePx)
    val iconCanvas = Canvas(iconBitmap)
    iconDrawable.setBounds(0, 0, sizePx, sizePx)
    iconDrawable.draw(iconCanvas)

    // Create a new bitmap for the final marker
    val outputBitmap = createBitmap(sizePx, sizePx)
    val canvas = Canvas(outputBitmap)

    // Draw filled circle
    val paint = Paint(Paint.ANTI_ALIAS_FLAG)
    paint.color = backgroundColor
    paint.style = Paint.Style.FILL
    val radius = sizePx / 2f
    canvas.drawCircle(radius, radius, radius, paint)

    // Draw black circumference
    paint.color = Color.BLACK
    paint.style = Paint.Style.STROKE
    paint.strokeWidth = sizePx * 0.04f // 8% of size as border thickness
    canvas.drawCircle(radius, radius, radius - paint.strokeWidth / 2, paint)

    // Draw the icon centered
    val foregroundPaint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        colorFilter = PorterDuffColorFilter(foregroundColor, PorterDuff.Mode.SRC_IN)
    }
    /*val foregroundPaint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        Paint.setColorFilter = PorterDuffColorFilter(foregroundColor, PorterDuff.Mode.SRC_IN)
    }*/
    // Draw the icon centered and tinted
    val iconInset = (sizePx * 0.18f).toInt()
    val iconRect = Rect(iconInset, iconInset, sizePx - iconInset, sizePx - iconInset)
    canvas.drawBitmap(iconBitmap, null, iconRect, foregroundPaint)

    return outputBitmap //BitmapDrawable(context.resources, outputBitmap)
}

//selectedUrl = "file:///android_asset/light-v10.json"
fun getMapboxUrl(styleIndex: Int, token: String, context: Context): String {
    val url = tileServers[MAPBOX_TILE_SERVER].url
    val style = try { mapboxStyles[styleIndex] }
    catch ( _:Exception) { mapboxStyles[mapboxStyles.size-1].lowercase() }

    // Style doesn't hold the token, inject it now
    val styleJson = context.assets.open("$url/$style.json")
        .bufferedReader()
        .use { it.readText() }
    //val updatedJson = styleJson.replace("__ACCESS_TOKEN__", token)
    //val tempFile = File(context.cacheDir, "mapboxStyle.json")
    //tempFile.writeText(updatedJson)

    return styleJson.replace("__ACCESS_TOKEN__", token) //tempFile.absolutePath //"$url$style.json?rapidapi-key=$apiKey".lowercase()

    //return "$url$style.json"
}

fun getAvailableMapboxStyles(context: Context): List<String> {
    val assetManager = context.assets
    val styleFolder = "mapbox_styles"

    return assetManager.list(styleFolder)
        ?.filter { it.endsWith(".json") }
        ?.map { it.removeSuffix(".json") }  // Convert back to style name
        ?: emptyList()
}

fun getMapToolkitUrl(styleIndex: Int, apiKey: String): String {
    val url = tileServers[MAPTOOLKIT_TILE_SERVER].url
    val style = try { mapToolKitStyles[styleIndex] }
    catch ( _:Exception) { mapToolKitStyles[DEFAULT_MAPTOOLKIT_STYLE].lowercase() } // Last one is Street

    return "$url$style.json?rapidapi-key=$apiKey".lowercase()
}

fun getOpenFreeMapUrl(styleIndex: Int): String {
    val url = tileServers[OPENFREEMAP_TILE_SERVER].url
    val style = try { openFreeMapStyles[styleIndex] }
    catch ( _:Exception) { openFreeMapStyles[DEFAULT_OPENFREEMAP_STYLE].lowercase() }

    return "$url${style.lowercase()}"
}

// --- Compute distance between two LatLng points in meters ---
fun distanceBetween(a: LatLng, b: LatLng): Double {
    val R = 6371000.0 // Earth radius in meters
    val dLat = Math.toRadians(b.latitude - a.latitude)
    val dLon = Math.toRadians(b.longitude - a.longitude)
    val lat1 = Math.toRadians(a.latitude)
    val lat2 = Math.toRadians(b.latitude)

    val haversine = sin(dLat / 2).pow(2.0) +
            sin(dLon / 2).pow(2.0) * cos(lat1) * cos(lat2)

    return 2 * R * asin(sqrt(haversine))
}

// --- Compute appropriate animation duration in milliseconds ---
fun calculateAnimationDuration(current: LatLng, target: LatLng): Int {
    val minDuration = 300     // ms
    val maxDuration = 3000    // ms

    val distance = distanceBetween(current, target)
    val computed = (distance * 0.5).toInt() // 0.5 ms per meter, adjust as needed

    return computed.coerceIn(minDuration, maxDuration)
}

fun distanceInMeters(a: LatLng, b: LatLng): Double {
    val results = FloatArray(1)
    Location.distanceBetween(a.latitude, a.longitude, b.latitude, b.longitude, results)
    return results[0].toDouble()
}

@Composable
fun MapAttributionOverlay(
    modifier: Modifier = Modifier,
    selectedTileServer: Int,
    showMapToolkit: Boolean = false
) {
    var showDialog by remember { mutableStateOf(false) }
    val context = LocalContext.current

    Box(modifier = modifier.fillMaxSize()) {
        // Bottom row: logos and attribution text
        Row(modifier = Modifier.align(Alignment.BottomStart).padding(8.dp),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.spacedBy(6.dp)) {
            when (selectedTileServer) {
                MAPBOX_TILE_SERVER -> Image(painter = painterResource(id = R.drawable.mapbox_logo), contentDescription = "Map Provider Logo", modifier = Modifier.height(20.dp))
                MAPTOOLKIT_TILE_SERVER -> Text("© $maptoolkitName", style = MaterialTheme.typography.bodySmall.copy(color = MaterialTheme.colorScheme.onBackground.copy(alpha = 0.8f)))
                else -> Text("© OpenStreetMap", style = MaterialTheme.typography.bodySmall.copy(color = MaterialTheme.colorScheme.onBackground.copy(alpha = 0.8f)))
            }
            IconButton(onClick = { showDialog = true }, modifier = Modifier.size(24.dp)) {
                Icon(Icons.Default.Info, contentDescription = "Attribution info", tint = MaterialTheme.colorScheme.onBackground.copy(alpha = 0.6f))
            }
        }
        // Dialog with clickable attribution links
        if (showDialog) {
            AlertDialog(
                onDismissRequest = { showDialog = false },
                confirmButton = { Button(onClick = { showDialog = false }) { Text(stringResource( android.R.string.ok)) } },
                title = { Text(stringResource(R.string.attribution), style = MaterialTheme.typography.titleMedium) },
                text = {
                    Column(
                        modifier = Modifier
                            .fillMaxWidth()
                            .verticalScroll(rememberScrollState())
                    ) {
                        val links = mutableListOf<Pair<String, String>>()
                        links += "MapLibre Android" to "https://maplibre.org/"
                        links += "© OpenStreetMap" to "https://www.openstreetmap.org/copyright"

                        if (selectedTileServer == MAPBOX_TILE_SERVER) {
                            links += "© Mapbox" to "https://www.mapbox.com/"
                            links += "Improve this map" to "https://apps.mapbox.com/feedback/"
                        }
                        if (selectedTileServer == MAPTOOLKIT_TILE_SERVER) links += "© $maptoolkitName" to "https://maptoolkit.com"

                        links.forEachIndexed { index, pair ->
                            Row(
                                verticalAlignment = Alignment.CenterVertically,
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .clickable { openUrl(context, pair.second) }
                                    .padding(vertical = 6.dp)
                            ) {
                                Icon(
                                    imageVector = Icons.AutoMirrored.Filled.OpenInNew,
                                    contentDescription = null,
                                    modifier = Modifier.size(18.dp),
                                    tint = MaterialTheme.colorScheme.primary
                                )
                                Spacer(modifier = Modifier.width(6.dp))
                                Text(
                                    text = pair.first,
                                    style = MaterialTheme.typography.bodyMedium.copy(
                                        color = MaterialTheme.colorScheme.primary
                                    )
                                )
                            }

                            if (index < links.lastIndex) Divider(modifier = Modifier.padding(vertical = 2.dp))
                        }
                    }
                },
                shape = RoundedCornerShape(12.dp),
                modifier = Modifier.padding(8.dp)
            )
        }
    }
}

private fun openUrl(context: Context, url: String) {
    val intent = Intent(Intent.ACTION_VIEW, url.toUri())
    context.startActivity(intent)
}
