package com.zell_mbc.publicartexplorer.maps

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.location.Location
import android.location.LocationManager
import androidx.core.content.ContextCompat
import androidx.core.location.LocationManagerCompat

fun locationEnabled(context: Context): Boolean {
    val locationManager = context.getSystemService(Context.LOCATION_SERVICE) as LocationManager
    return LocationManagerCompat.isLocationEnabled(locationManager)
}

// Lookup lastknown location and in case nothing is known fallback to default position
fun getLastKnownLocation(context: Context): Location? {
    // Check location permission
    val fineLocationGranted = ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED
    val coarseLocationGranted = ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION) == PackageManager.PERMISSION_GRANTED

    if (!fineLocationGranted && !coarseLocationGranted) {
        // No location permission granted
        return null
    }

    // LocationManager for FOSS flavour
    val locationManager = context.getSystemService(Context.LOCATION_SERVICE) as LocationManager

    // Get last known location from all providers, pick the best (most recent)
    val providers = locationManager.getProviders(true)
    var bestLocation: Location? = null
    for (provider in providers) {
        val location = try {
            // More robust provider check based on permissions
            if (provider == LocationManager.GPS_PROVIDER && !fineLocationGranted) {
                null // Don't try GPS if fine permission isn't granted
            } else if (provider == LocationManager.NETWORK_PROVIDER) {
                locationManager.getLastKnownLocation(provider)
            } else if (fineLocationGranted) { // For GPS_PROVIDER (if fine granted) and other fine providers
                locationManager.getLastKnownLocation(provider)
            }
            else {
                null
            }
        } catch (_: SecurityException) { // Catch SecurityException for LocationManager
            // Log.e("LocationError", "SecurityException with LocationManager for provider $provider", e)
            null
        }
        if (location != null) {
            if (bestLocation == null || location.time > bestLocation.time) {
                bestLocation = location
            }
        }
    }
    return bestLocation
}
