package com.zell_mbc.publicartexplorer.detailview

import PreferencesManager
import android.text.TextUtils.split
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.text.selection.SelectionContainer
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import com.zell_mbc.publicartexplorer.OSM_ACCESS_TOKEN_KEY
import com.zell_mbc.publicartexplorer.Artwork
import com.zell_mbc.publicartexplorer.ClickableUrlText
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.createClickableUrl
import com.zell_mbc.publicartexplorer.data.ViewModel
import com.zell_mbc.publicartexplorer.oauth2.TokenManager
import kotlin.String
import kotlin.collections.iterator
import kotlin.text.contains

@Composable
fun ShowRawData(viewModel: ViewModel, selectedArtwork: Artwork) {
    val context = LocalContext.current
    var showEditTags by remember { mutableStateOf<Boolean>( false) }

    val prefs = PreferencesManager(context)
    val token = TokenManager.osmToken.accessToken //by prefs.getStringFlow(OSM_ACCESS_TOKEN_KEY, "").collectAsState(initial = "")
    Box(modifier = Modifier
        //.fillMaxHeight()
        .fillMaxWidth()
        .padding(8.dp)) {
        SelectionContainer {
            Column(modifier = Modifier.fillMaxWidth().verticalScroll(rememberScrollState())) {

                val nodeId = "OSM id: " + selectedArtwork.id
                Text(
                    nodeId,
                    style = MaterialTheme.typography.bodyMedium,
                    color = MaterialTheme.colorScheme.onSurface
                )
                val coordinates = "coordinates: " + selectedArtwork.lat + ", " + selectedArtwork.lon
                Text(
                    coordinates,
                    style = MaterialTheme.typography.bodyMedium,
                    color = MaterialTheme.colorScheme.onSurface
                )

                for (tag in selectedArtwork.tags) {
                    val key = tag.key
                    val value = tag.value

                    if (value.contains("https://")) {
                        ClickableUrlText(
                            url = value,
                            textBefore = "$key=",
                            textStyle = MaterialTheme.typography.bodyMedium,
                            textColor = MaterialTheme.colorScheme.onSurface
                        )
                    } else when (key) { // Some known url structures
                        "subject:wikipedia" -> ClickableUrlText(
                            url = "https://wikipedia.org/wiki/$value",
                            urlText = value,
                            textBefore = "$key=",
                            textStyle = MaterialTheme.typography.bodyMedium,
                            textColor = MaterialTheme.colorScheme.onSurface
                        )
                        "subject:wikidata" -> ClickableUrlText(
                            url = "https://wikidata.org/wiki/$value",
                            urlText = value,
                            textBefore = "$key=",
                            textStyle = MaterialTheme.typography.bodyMedium,
                            textColor = MaterialTheme.colorScheme.onSurface
                        )
                        "artist:wikidata" -> ClickableUrlText(
                            url = "https://wikidata.org/wiki/$value",
                            urlText = value,
                            textBefore = "$key=",
                            textStyle = MaterialTheme.typography.bodyMedium,
                            textColor = MaterialTheme.colorScheme.onSurface
                        )
                        "artist:wikipedia" -> ClickableUrlText(
                            url = "https://wikipedia.org/wiki/$value",
                            urlText = value,
                            textBefore = "$key=",
                            textStyle = MaterialTheme.typography.bodyMedium,
                            textColor = MaterialTheme.colorScheme.onSurface
                        )
                        "wikimedia_commons" -> ClickableUrlText(
                            url = "https://commons.wikimedia.org/wiki/$value",
                            urlText = value,
                            textBefore = "$key=",
                            textStyle = MaterialTheme.typography.bodyMedium,
                            textColor = MaterialTheme.colorScheme.onSurface
                        )
                        else -> Text(
                            text = "$key=$value",
                            fontWeight = FontWeight.Normal,
                            style = MaterialTheme.typography.bodyMedium,
                            color = MaterialTheme.colorScheme.onSurface
                        )
                    }
                }
                // If this node is part of a relation show the relation specific data now
                ShowRelationsRawData(viewModel, selectedArtwork.id)
                Text("")
                HorizontalDivider(thickness = 1.dp, color = MaterialTheme.colorScheme.primary)
                Text("")

                Row {
                    Text(context.getString(R.string.source) + ": ")
                    // Format: /?mlat=LAT&mlon=LON#map=ZOOM/LAT/LON
                    // Example: https://www.openstreetmap.org/?mlat=42.12345&mlon=12.54321#map=18/42.12345/12.54321
                    Text(
                        createClickableUrl(
                            "https://www.openstreetmap.org/?mlat=${selectedArtwork.lat}&mlon=${selectedArtwork.lon}#map=18/${selectedArtwork.lat}/${selectedArtwork.lon}" + selectedArtwork.id,
                            "OpenStreetMap"
                        )
                    )
                }
                // Show only if a osm token is present
                if (!token.isNullOrEmpty())
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        horizontalArrangement = Arrangement.End
                    ) {
                        OutlinedButton(onClick = { showEditTags = true }) {
                            Text(
                                stringResource(R.string.editTags),
                                textAlign = TextAlign.End,
                                color = MaterialTheme.colorScheme.primary
                            )
                        }
                    }
            }
        }
        if (showEditTags) {
            val originalTags = selectedArtwork.tags
            EditTagsDialog(viewModel, stringResource(R.string.editArtwork), tags =selectedArtwork.tags,
                onDismiss = { showEditTags = false },
                onConfirm = {
                    // Check if anything changed
                    val changedEntries = (it.filter { (k, v) -> originalTags[k] != v }).count()
                    if (changedEntries > 0) {
                        selectedArtwork.tags = LinkedHashMap(it) // Convert from state to actual value to make sure it survives
                        viewModel.updateOsm(selectedArtwork, token ?: "")
                        viewModel.fetchImages() // In case images got added/changed/removed
                    }
                    showEditTags = false
                },
            )
        }
    }
}


