package com.zell_mbc.publicartexplorer.detailview

import android.content.Intent
import android.net.Uri
import android.util.Log
import android.widget.Toast
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Upload
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.MenuAnchorType
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.DialogProperties
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.data.ViewModel
import com.zell_mbc.publicartexplorer.helpers.normalizeCommonsUrl
import com.zell_mbc.publicartexplorer.oauth2.TokenManager
import com.zell_mbc.publicartexplorer.screens.LocalSnackbarHostState
import com.zell_mbc.publicartexplorer.wiki.MediaStorePickerDialog
import com.zell_mbc.publicartexplorer.wiki.extractGpsFromImageUri
import com.zell_mbc.publicartexplorer.wiki.fetchCategories
import com.zell_mbc.publicartexplorer.wiki.rememberGpsSafeImagePicker
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlin.collections.component1
import kotlin.collections.component2

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun EditTagsDialog(viewModel: ViewModel, title: String,
                   onDismiss: () -> Unit,
                   onConfirm: (MutableMap<String, String>) -> Unit,
                   tags: MutableMap<String, String>) {
    val context = LocalContext.current
    val snackbarHostState = LocalSnackbarHostState.current
    val scope = rememberCoroutineScope()

    var showWarningDialog by remember { mutableStateOf(false) }
    var showAddTagDialog by remember { mutableStateOf(false) }
    var showImagePicker by remember { mutableStateOf(false) }
    var changedEntries by remember { mutableIntStateOf(0) }

    // Remember state for all fields and load only once to avoid a reset during orientation changes
    if (viewModel.currentTags.isEmpty())
        viewModel.currentTags.apply { putAll(tags) }

    val originalTags = tags

    // Derived state that is true if there are changes
    val hasChanges by remember { derivedStateOf { viewModel.currentTags.any { (k, v) -> originalTags[k] != v } || originalTags.keys.any { it !in viewModel.currentTags.keys } } }
    var imageUploadInProgress by remember { mutableStateOf (false) }
    var pickedImageUri by remember { mutableStateOf<Uri?>(null) }
    var pickedImageUriHasGps by remember { mutableStateOf(false) }
    val accessToken = TokenManager.wikiToken.accessToken ?: ""
    var suggestedCategories by remember { mutableStateOf<List<String>>(emptyList()) }
    val selectedCategories by viewModel.selectedCategories
    var tempSelectedCategories by remember { mutableStateOf<List<String>>(emptyList()) } // Holds values during category entry
    var showCategoryDialog by remember { mutableStateOf(false) }

    LaunchedEffect(pickedImageUri) {
        if (pickedImageUri != null) {

            val location = extractGpsFromImageUri(context, pickedImageUri!!)
            if (location.longitude != null && location.latitude != null) {
                CoroutineScope(Dispatchers.IO).launch {
                    suggestedCategories = fetchCategories(location)
                }
            }

            showCategoryDialog = true
            Log.d("GPS", "Lat: ${location.latitude}, Lng: ${location.longitude}")
        }
    }

    val uploadState = viewModel.uploadState

    when (uploadState) {
        is ViewModel.UploadState.Uploading -> {
            viewModel.currentTags["wikimedia_commons"] = "Wikimedia upload in progress..."
            imageUploadInProgress = true
        }
        is ViewModel.UploadState.Success -> {
            viewModel.currentTags["wikimedia_commons"] = "File:" + (uploadState as ViewModel.UploadState.Success).filename
            imageUploadInProgress = false
        }
        is ViewModel.UploadState.Error -> viewModel.currentTags["wikimedia_commons"] = ""
        else -> { /* Idle, show nothing */ }
    }

    LaunchedEffect(selectedCategories) {
        if (selectedCategories.isNotEmpty()) {
            viewModel.uploadWikidataImage(context = context, accessToken =accessToken, uri = pickedImageUri, categories = selectedCategories )
        }
    }

    if (showCategoryDialog) {
        if (suggestedCategories.isEmpty()) suggestedCategories += "Public art in…"
        AlertDialog(
            onDismissRequest = { showCategoryDialog = false },
            title = { Text("Select Categories") },
            text = {
                WikiCategorySelector(
                    suggestedCategories = suggestedCategories,
                    initialSelection = tempSelectedCategories,
                    hasGps = pickedImageUriHasGps,
                    onSelectionChanged = { selected -> tempSelectedCategories = selected }
                )
            },
            confirmButton = { Button(onClick = {
                showCategoryDialog = false
                viewModel.selectedCategories.value = tempSelectedCategories
            }) { Text(stringResource(R.string.ok)) } },
            dismissButton = {
                OutlinedButton(onClick = { showCategoryDialog = false }) {
                    Text(stringResource(R.string.cancel))
                }
            }
        )
    }

    val imagePicker = rememberGpsSafeImagePicker(context) { uri, location ->
        pickedImageUri = uri
        location?.let {
            pickedImageUriHasGps = (location.latitude != 0.0)
            Log.d("ImagePicker", "Latitude: ${it.latitude}, Longitude: ${it.longitude}")
        }
    }

    AlertDialog(
        onDismissRequest = { /* Do nothing */ },
        properties = DialogProperties(
            usePlatformDefaultWidth = false, // allows more flexible layout
            dismissOnBackPress = true,
            dismissOnClickOutside = false
        ),
        title = {
            Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.SpaceBetween, verticalAlignment = Alignment.CenterVertically) {
                Text(text = title, style = MaterialTheme.typography.titleLarge)
                Icon(
                    modifier = Modifier.size(32.dp),
                    painter = painterResource(R.drawable.icon_artwork),
                    contentDescription = "Sculpture Icon"
                )
            }
        },
        text = {
            LazyColumn(modifier = Modifier.fillMaxWidth() //.fillMaxHeight(0.8f) // make room for keyboard
                    .imePadding()        // adjust for keyboard
            ) {
                items(viewModel.currentTags.entries.toList()) { (key, value) ->
                    when (key) {
                        "tourism" -> { /* Ignore */ }
                        "artwork_type" -> ArtworkType(value) { newValue -> viewModel.currentTags[key] = newValue }
                        "material" -> Material(value) { newValue -> viewModel.currentTags[key] = newValue }
                        "wheelchair" -> Wheelchair(value) { newValue -> viewModel.currentTags[key] = newValue }
                        "wikimedia_commons" -> OutlinedTextField(
                            value = value,
                            onValueChange = { newValue ->
                                viewModel.currentTags[key] = if (newValue.startsWith("https://commons.wikimedia.org/wiki/")) normalizeCommonsUrl(newValue)
                                else newValue
                                },
                            label = { Text("$key image") },
                            trailingIcon = {
                                IconButton(
                                    //enabled = !TokenManager.wikiToken.isExpired(),
                                    onClick = {
                                        if (TokenManager.wikiToken.isExpired()) { // Show logon Scnackbar instead
                                            Toast.makeText(context, context.getString(R.string.wikiNotSignedIn), Toast.LENGTH_SHORT).show()
                                            /* scope.launch {
                                                val result = snackbarHostState.showSnackbar(
                                                    message = context.getString(R.string.wikiNotSignedIn),
                                                    actionLabel = context.getString(R.string.login),
                                                    duration = SnackbarDuration.Short
                                                )
                                                when (result) {
                                                    SnackbarResult.ActionPerformed -> startWikiLogin(context)
                                                    SnackbarResult.Dismissed -> { /* Do nothing */
                                                    }
                                                }
                                            }*/
                                            }
                                        else {
                                            // In case we start over
                                            pickedImageUri = null
                                            tempSelectedCategories = selectedCategories
                                            suggestedCategories = emptyList()

                                            // 💀💢😡🔥😤💥
                                            // https://issuetracker.google.com/issues/243294058
                                            val bugfixImagePicker = false
                                            if (bugfixImagePicker)
                                                showImagePicker = true
                                            else {
                                                val intent = Intent(Intent.ACTION_OPEN_DOCUMENT).apply {
                                                    addCategory(Intent.CATEGORY_OPENABLE)
                                                    type = "image/*"
                                                }
                                                imagePicker.launch(intent)
                                            }
                                        }
                                    }
                                ) {
                                    Icon(Icons.Default.Upload, contentDescription = "Upload image")
                                }
                            },
                            modifier = Modifier.fillMaxWidth()
                        )
                        else -> OutlinedTextField(
                            value = value,
                            onValueChange = { newValue -> viewModel.currentTags[key] = newValue },
                            label = { Text(key) },
                            modifier = Modifier.fillMaxWidth()
                        )
                    }
                    Spacer(modifier = Modifier.height(8.dp))
                }
            }
        },
    dismissButton = {
        Column(modifier = Modifier.fillMaxWidth()) {
            // Row 1: Add tag, right aligned
            Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.End) {
                OutlinedButton(onClick = { showAddTagDialog = true }) { Text(stringResource(R.string.addTag)) }
            }
            Spacer(modifier = Modifier.height(8.dp))

            // Row 2: Cancel + Upload, right aligned
            Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.End) {
                // Cancel
                OutlinedButton(onClick = {
                    viewModel.currentTags.clear()
                    onDismiss()
                }) { Text(stringResource(R.string.cancel)) }
                Spacer(modifier = Modifier.width(8.dp))
                // Upload
                Button(enabled = hasChanges && !imageUploadInProgress,
                    onClick = {
                        fun norm(s: String?) = s?.trim() ?: ""

                        // Todo: What are the minimum tags required?

                        // Check what changed
                        val added = (viewModel.currentTags.keys - originalTags.keys).size
                        val removed = (originalTags.keys - viewModel.currentTags.keys).size
                        val modified = viewModel.currentTags.keys
                            .intersect(originalTags.keys)
                            .count { k -> norm(viewModel.currentTags[k]) != norm(originalTags[k]) }

                            changedEntries = added + removed + modified
                            if (changedEntries > 0) {
                                showWarningDialog = true
                            } else {
                              viewModel.currentTags.clear()
                              onDismiss()
                          }
                      }
                ) { Text(stringResource(R.string.upload)) }
            }
        }
    },
    // Empty confirmButton to avoid double padding
    confirmButton = {}
    )

    if (showWarningDialog) {
        val title = "⚠️ " + stringResource(R.string.warning) + if (viewModel.sandboxMode) " Sandbox mode on!" else ""
        UploadWarningDialog(
            title = title,
            updates = changedEntries,
            showDialog = showWarningDialog,
            onDismiss = { showWarningDialog = false },
            onConfirm = {
                showWarningDialog = false
                onConfirm(viewModel.currentTags) // 🚀 perform upload here
                viewModel.currentTags.clear() // Todo: Check if values are still there in onConfirm
            }
        )
    }

    if (showAddTagDialog)
        AddTagDialog(
            onDismiss = { showAddTagDialog = false },
            onConfirm = { key -> viewModel.currentTags[key] = "" // Add new entry
            }
        )

    if (showImagePicker) {
        MediaStorePickerDialog(
            show = showImagePicker,
            onDismiss = { showImagePicker = false },
            onImagePicked = { uri ->
                showImagePicker = false
                pickedImageUri = uri
                val location = extractGpsFromImageUri(context, uri)
                Log.d("Picker", "Lat=${location?.latitude}, Lon=${location?.longitude}")
            }
        )    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ArtworkType(currentValue: String, onChange: (String) -> Unit ) {
    var artworkTypeExpanded by remember { mutableStateOf(false) }
    var artworkType by remember { mutableStateOf(currentValue) }

    val artworkTypeOptions = listOf("stone","architecture","bust","installation","relief","graffiti", "statue","sculpture", "mural", "fountain", "monument") // OSM-known types

    // Artwork Type dropdown
    ExposedDropdownMenuBox(expanded = artworkTypeExpanded, onExpandedChange = { artworkTypeExpanded = !artworkTypeExpanded }) {
        OutlinedTextField(value = artworkType, onValueChange = {}, readOnly = true, label = { Text("artwork_type") },
            modifier = Modifier.menuAnchor(type = MenuAnchorType.PrimaryEditable, enabled = true).fillMaxWidth(),
            trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded = artworkTypeExpanded) }
        )
        ExposedDropdownMenu(expanded = artworkTypeExpanded, onDismissRequest = { artworkTypeExpanded = false }) {
            artworkTypeOptions.forEach { type ->
                DropdownMenuItem(text = { Text(type) },
                    onClick = {
                        artworkType = type
                        onChange(type)
                        artworkTypeExpanded = false
                    })
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun Material(currentValue: String, onChange: (String) -> Unit) {
    var materialExpanded by remember { mutableStateOf(false) }
    var material by remember { mutableStateOf(currentValue) }
    val materialOptions = listOf("stone", "metal", "wood", "concrete","plastic")

    // Material dropdown (editable combo box)
    ExposedDropdownMenuBox(expanded = materialExpanded, onExpandedChange = { materialExpanded = !materialExpanded }) {
        OutlinedTextField(value = material,
            onValueChange = { onChange(it) },   // ✅ allow typing
            readOnly = false,                    // ✅ make editable
            label = { Text("material") }, modifier = Modifier
            .menuAnchor(type = MenuAnchorType.PrimaryEditable, enabled = true).fillMaxWidth(),
            trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded = materialExpanded) }
        )
        ExposedDropdownMenu(expanded = materialExpanded, onDismissRequest = { materialExpanded = false }) {
            materialOptions.forEach { mat ->
                DropdownMenuItem(text = { Text(mat) },
                    onClick = {
                        material = mat
                        materialExpanded = false
                        onChange(mat)
                    })
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun Wheelchair(currentValue: String, onChange: (String) -> Unit) {
    var wheelchairExpanded by remember { mutableStateOf(false) }
    val wheelchairOptions = listOf("yes", "no", "limited")

    // Wheelchair accessibility
    ExposedDropdownMenuBox(expanded = wheelchairExpanded, onExpandedChange = { wheelchairExpanded = !wheelchairExpanded }) {
        OutlinedTextField(value = currentValue, onValueChange = {}, readOnly = true, label = { Text("wheelchair") },
            modifier = Modifier.menuAnchor(type = MenuAnchorType.PrimaryEditable, enabled = true).fillMaxWidth(),
            trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded = wheelchairExpanded) }
        )
        ExposedDropdownMenu(expanded = wheelchairExpanded, onDismissRequest = { wheelchairExpanded = false }) {
            wheelchairOptions.forEach { option ->
                DropdownMenuItem(text = { Text(option) },
                    onClick = {
                        onChange(option)
                        wheelchairExpanded = false
                    }
                )
            }
        }
    }
}

@Composable
fun UploadWarningDialog(title: String, updates: Int, showDialog: Boolean,
    onDismiss: () -> Unit, onConfirm: () -> Unit) {
    if (showDialog) {
        AlertDialog(
            onDismissRequest = onDismiss,
            title = { Text(text = title, style = MaterialTheme.typography.titleLarge, color = MaterialTheme.colorScheme.error) },
            text = { Text(stringResource(R.string.caution, updates)) },
            confirmButton = {
                Button(onClick = onConfirm) {
                    Text(stringResource(R.string.proceed))
                }
            },
            dismissButton = {
                OutlinedButton(onClick = onDismiss) {
                Text(stringResource(R.string.cancel))
                }
            }
        )
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun AddTagDialog(onDismiss: () -> Unit, onConfirm: (String) -> Unit) {
    val availableTags = listOf(
        "artist:wikidata" to "Wikidata entity ID of the artist",
        "artist:wikipedia" to "Wikipedia link to the artist",
        "artist_name" to "Name of the artist",
        "artwork_type" to "Type of artwork (sculpture, mural, etc.)",
        "description" to "Some explanatory description",
        "name" to "Name of the artwork",
        "wikimedia_commons" to "Image on Wikimedia Commons",
        "panoramax" to "Panoramax image reference",
        "image" to "Image on different location",
        "start_date" to "Date when the artwork was installed",
        "subject:wikidata" to "Wikidata tag of the subject, usually starts with Q",
        "subject:wikipedia" to "Wikipedia link to the subject",
        "wheelchair" to "Wheelchair accessibility"
    )
    var expanded by remember { mutableStateOf(false) }
    var selectedTag by remember { mutableStateOf("") }

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(stringResource(R.string.addTag)) },
        text = {
            ExposedDropdownMenuBox(
                expanded = expanded,
                onExpandedChange = { expanded = !expanded }
            ) {
        OutlinedTextField(
            value = selectedTag,
            onValueChange = {},
            readOnly = true,
            label = { Text("Select tag") },
            trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded) },
            modifier = Modifier.menuAnchor()
        )

        ExposedDropdownMenu(
            expanded = expanded,
            onDismissRequest = { expanded = false }
            ) {
            availableTags.forEach { (tag, explanation) ->
            DropdownMenuItem(
                text = {
                        Column {
                        Text(tag, style = MaterialTheme.typography.bodyMedium)
                        Text(text = explanation, style = MaterialTheme.typography.bodySmall, color = MaterialTheme.colorScheme.onSurfaceVariant)
                  }
                },
                onClick = {
                  selectedTag = tag
                  expanded = false
                }
            )
        }
        }
    }
    },
    confirmButton = {
        Button(
            onClick = {
                if (selectedTag.isNotBlank()) {
                onConfirm(selectedTag.trim())
                onDismiss() // close the dialog
            }
        }) { Text(stringResource(R.string.add)) }
    },
    dismissButton = {
        OutlinedButton(onClick = onDismiss) {
        Text(stringResource(R.string.cancel))
        }
    }
    )
}
