package com.zell_mbc.publicartexplorer.data

import android.app.Application
import android.content.Context
import android.net.Uri
import android.util.Log
import android.widget.Toast
import androidx.lifecycle.AndroidViewModel
import com.zell_mbc.publicartexplorer.Artwork
import com.zell_mbc.publicartexplorer.NodeType
import com.zell_mbc.publicartexplorer.R
import com.zell_mbc.publicartexplorer.Relation
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.withContext
import org.json.JSONArray
import org.json.JSONObject
import java.io.IOException
import kotlin.collections.listOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableLongStateOf
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateMapOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.content.edit
import androidx.lifecycle.application
import androidx.lifecycle.viewModelScope
import com.zell_mbc.publicartexplorer.Constants
import com.zell_mbc.publicartexplorer.DARK_MODE_KEY
import com.zell_mbc.publicartexplorer.DEFAULT_POI_SERVER
import com.zell_mbc.publicartexplorer.DarkThemeOptions
import com.zell_mbc.publicartexplorer.DebugLog
import com.zell_mbc.publicartexplorer.ENABLE_HAPTIC_FEEDBACK_KEY
import com.zell_mbc.publicartexplorer.EXPERT_MODE_KEY
import com.zell_mbc.publicartexplorer.MAPBOX_TILE_SERVER
import com.zell_mbc.publicartexplorer.MAP_STYLE_KEY
import com.zell_mbc.publicartexplorer.OPENFREEMAP_TILE_SERVER
import com.zell_mbc.publicartexplorer.POI_SERVER_KEY
import com.zell_mbc.publicartexplorer.RADIUS_KEY
import com.zell_mbc.publicartexplorer.SHOW_MUSEUM_KEY
import com.zell_mbc.publicartexplorer.SUBSCRIPTION_ACTIVE_KEY
import com.zell_mbc.publicartexplorer.TILE_SERVER_KEY
import com.zell_mbc.publicartexplorer.billing.BillingManager
import com.zell_mbc.publicartexplorer.billing.BillingManagerImpl
import com.zell_mbc.publicartexplorer.detailview.ImageItem
import com.zell_mbc.publicartexplorer.getHttpErrorMessage
import com.zell_mbc.publicartexplorer.isPhotoAlbum
import com.zell_mbc.publicartexplorer.mapboxStyles
import com.zell_mbc.publicartexplorer.nanoToTime
import com.zell_mbc.publicartexplorer.osm.createDataset
import com.zell_mbc.publicartexplorer.wiki.uploadImageToCommons
import com.zell_mbc.publicartexplorer.poiServers
import com.zell_mbc.publicartexplorer.sharedOkHttpClient
import com.zell_mbc.publicartexplorer.wiki.getCsrfToken
import io.github.dellisd.spatialk.geojson.Position
import kotlinx.coroutines.launch
import okhttp3.Request
import okhttp3.Response
import kotlin.system.measureNanoTime


class ViewModel(application: Application) : AndroidViewModel(application) {
    val debugList = DebugLog.messages
    var debugMode = false

    // Billing stuff for Google Play
    val billingManager = BillingManagerProvider.instance     // Billing/Licensing stuff, only relevant for GooglePlay version, empty for FOSS
    object BillingManagerProvider { val instance: BillingManager by lazy { BillingManagerImpl() } }     // Use a singleton to avoid recreating every time MainActivity get's called

    // Datastore preferences
    //val prefs = PreferencesManager(application)
   // fun setPreference(preference: String, value: String) = viewModelScope.launch { prefs[preference] = value }
    //fun getPreferenceState(preference: String, default: String = "") = prefs.flow(preference, default)
    //suspend fun getPreferenceValue(preference: String, default: String = ""): String { return prefs.flow(preference, default).first() }

    // Preferences
    val preferences = application.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)

    // Load preferences as state variables
    var darkMode by mutableIntStateOf(preferences.getInt(DARK_MODE_KEY, DarkThemeOptions.FollowSystem.ordinal))
    var subscriptionActive by mutableStateOf(preferences.getBoolean(SUBSCRIPTION_ACTIVE_KEY, false))
    var expiryTimestamp by mutableLongStateOf(0L)
    var radius by mutableIntStateOf(preferences.getInt(RADIUS_KEY, Constants.DEFAULT_RADIUS))
    var expertMode by mutableStateOf(preferences.getBoolean(EXPERT_MODE_KEY, Constants.DEFAULT_EXPERT_MODE))
    var sandboxMode by mutableStateOf(false)  // If true, node changes are not uploaded to OSM
    var selectedMapStyleIndex by mutableStateOf(preferences.getInt(MAP_STYLE_KEY, 0))
    var selectedPoiServer by mutableStateOf(preferences.getString(POI_SERVER_KEY, poiServers[DEFAULT_POI_SERVER].name ))
    var selectedTileServer by mutableIntStateOf(preferences.getInt(TILE_SERVER_KEY, OPENFREEMAP_TILE_SERVER))
    var showMuseum by mutableStateOf(preferences.getBoolean(SHOW_MUSEUM_KEY, Constants.DEFAULT_SHOW_MUSEUM))
    //var openAppAtCurrentPosition by mutableStateOf(preferences.getBoolean(LAUNCH_AT_CURRENT_POSITION_KEY, Constants.DEFAULT_LAUNCH_AT_CURRENT_POSITION))
    //var zoomLevelFiltering by mutableStateOf(preferences.getBoolean(ZOOM_LEVEL_FILTERING_KEY, Constants.DEFAULT_ZOOM_LEVEL_FILTERING))
    val enableHapticFeedback by mutableStateOf(preferences.getBoolean(ENABLE_HAPTIC_FEEDBACK_KEY, true))
    var showSearchCircle by mutableStateOf(false)

    //var artworkGeoJsonData by remember { mutableStateOf<JSONObject?>(null) }
    var selectedArtwork by mutableStateOf<Artwork?>(null)
    var selectedPosition by mutableStateOf<Position?>(null) // Filled when user long presses on the map
    var showRawData by mutableStateOf(false)
    var showAddressSearch by mutableStateOf(false)
    var showNewNodeDialog by mutableStateOf(false) // Handle the new node alert dialog
    var userLocation by mutableStateOf<Position?>(null) // Location is "physical"

    var findMe by mutableStateOf(false) // Show indicator while search for position is on
    var followMe by mutableStateOf(false) // Session setting instead of preference to avoid GPS to stay on forever preferences.getBoolean(FOLLOW_ME_KEY, Constants.DEFAULT_FOLLOW_ME))
    var gpsFixAvailable by mutableStateOf(false)

    // Var's driving screen behaviour
    var showThankYouDialogState by mutableStateOf(false)
    var showBillingUnavailableDialogState by mutableStateOf(false)
    var showUpgradeDialogState by mutableStateOf(false) // Do not delete, required for Play flavour

    var artworks = mutableListOf<Artwork>()
    var artworkCount = mutableStateOf(0)
    var relations = mutableListOf<Relation>()
    var artworksGeoJson: JSONObject? = null
    var imageItems = mutableStateListOf<ImageItem>()

    // Values used in dialogs
    var currentTags = mutableStateMapOf<String, String>() // Edit tags dialog
    var selectedCategories = mutableStateOf<List<String>>(emptyList())

    fun selectArtwork(id: Int) { selectedArtwork = artworks[id] }
    private fun clearImages() { imageItems.clear() }

    // Identify potential image provider and retrieve the associated urls
    fun fetchImages() {
        var artwork: Artwork

        if (selectedArtwork == null) return
        else artwork = selectedArtwork!!

        // Clean up list
        clearImages()

        // - Regular urls: https://upload.wikimedia.org/.../Bildstock.jpg
        // - Image viewer urls: https://commons.wikimedia.org/wiki/File:Bildstock_Unteropfingen_(60509).jpg
        val imageUrl = artwork.tags["image"] ?: ""
        if (imageUrl.isNotEmpty()) {
            if (isPhotoAlbum(imageUrl) == null) {
                addImageUrl(imageUrl)
            }
        }

        // Panoramax
        val panoramaxTags: List<String> = artwork.tags.entries
            .filter { it.key == "panoramax" || it.key.startsWith("panoramax:") }
            .map { it.value }  // extract the values

        // Example: iterate over all
        for (id in panoramaxTags) {
            addPanoramaxImage(id)
        }

        // Wikimedia Commons
        //Input: wikimedia_commons=File:Tony Stadler Skulptur Quellnymphe Ulm (1982) 02.jpg -> Söflingen
        //Input: File:Museum_Biberach,_Statue_von_Johann_Baptist_Pflug.jpg
        //Input: Sämtliche images in Unterwachingen
        val wikimediaCommons = artwork.tags["wikimedia_commons"] ?: ""
        if (wikimediaCommons.isNotEmpty()) {
            // Needs lookup of image first
            if (wikimediaCommons.contains("Category:")) { // Not a picture but a URL
                addWikimediaCommonsCategoryImage(wikimediaCommons)
            }
            else
            //In:  wikimedia_commons=File:Tony Stadler Skulptur Quellnymphe Ulm (1982) 02.jpg
            //Out: https://upload.wikimedia.org/wikipedia/commons/5/57/Toni_Stadler_Skulptur_Quellnymphe_Ulm_%281982%29_02.jpg
                // Add provider and : so addImageUrl can properly handle the tag
                addImageUrl("wikimedia:$wikimediaCommons") // Can be consumed directly
        }

        // Mapillary
        // Alphorn in Ochsenhausen
        val mapillaryId = artwork.tags["mapillary"] ?: ""
        if (mapillaryId.isNotEmpty())
            addMapillaryImage(id = mapillaryId)
        // }

        // WikiData tags
        // No filenames but tags, starting with "Q", E.g. "Q798798"
        // Ochsenhausen Klostermuseum: Q1776456
        val wikidataTag = artwork.tags["wikidata"] ?: ""
        if (wikidataTag.isNotEmpty()) addWikidataImage(wikidataTag)
    }

    fun addPanoramaxImage(id: String) {
        viewModelScope.launch {
            val item = fetchPanoramaxImage(id, application)
            item?.let { imageItems.add(it) }
        }
    }
    fun addImageUrl(id: String) {
        viewModelScope.launch {
            val item = fetchImageTag(id, application)
            item?.let { imageItems.add(it) }
        }
    }

    fun addWikimediaCommonsCategoryImage(category: String){
        viewModelScope.launch {
            val item = fetchWikimediaCommonsCategoryImage(category, application)
            item?.let { imageItems.add(it) }
        }
    }

    fun addWikidataImage(id: String){
        viewModelScope.launch {
            val item = fetchWikidataImage(id, application)
            item?.let { imageItems.add(it) }
        }
    }

    fun addMapillaryImage(id: String){
        viewModelScope.launch {
            val item = fetchMapillaryImage(id, application)
            item?.let { imageItems.add(it) }
        }
    }

    // Use MutableStateFlow or MutableLiveData for Compose-friendly observable state
    private val emptyGeoJsonString = """
    {
      "type": "FeatureCollection",
      "features": []
    }
    """.trimIndent()

    private val _artworkGeoJsonData = MutableStateFlow<JSONObject?>(JSONObject(emptyGeoJsonString))
    val artworkGeoJsonData: StateFlow<JSONObject?> = _artworkGeoJsonData.asStateFlow()

    fun setArtworkGeoJsonData(json: JSONObject?) {
        _artworkGeoJsonData.value = json
        DebugLog.add("_artworkGeoJsonData.value: " + _artworkGeoJsonData.value.toString())
    }


    init {
        // Callback functions! Not executed just yet, these get called from Billing manager
        billingManager.showBillingUnavailableDialog = { show -> showBillingUnavailableDialogState = show }
        billingManager.showThankYouDialog = { show ->
            showThankYouDialogState = show
            selectedTileServer = MAPBOX_TILE_SERVER
            selectedMapStyleIndex = mapboxStyles.size-1 // Will select the last one which is Streets
        }
        billingManager.setSubscriptionStatus = { value ->
            subscriptionActive = value
            // Make sure we go back to the free server
            if (!value){
                selectedTileServer = OPENFREEMAP_TILE_SERVER
                expertMode = false
                preferences.edit {
                    putBoolean(SUBSCRIPTION_ACTIVE_KEY, false)
                    putBoolean(EXPERT_MODE_KEY, false)
                }
            }
        }
        billingManager.setExpiryTimestamp = { value -> expiryTimestamp = value }

        // Check if this user has purchased a subscription
        billingManager.isSubscriptionActive(application) // This function will update ExpertMode and SUBSCRIPTION_ACTIVE_KEY asynchronously
    }

    // Fill artwork array and return geoString
    suspend fun fetchArtworks(context: Context, latitude: Double, longitude: Double): JSONObject? = withContext(Dispatchers.IO) {
        val preferences = context.getSharedPreferences("com.zell_mbc.publicartexplorer._preferences", Context.MODE_PRIVATE)
        val poiServerName = preferences.getString(POI_SERVER_KEY, poiServers[DEFAULT_POI_SERVER].name )
        var poiServer = poiServers.indexOfFirst { it.name == poiServerName }
        if (poiServer < 0) poiServer = 0
        val selectedPoiServer = poiServers[poiServer].url

        //val searchTerm = "\"archaeological_site\""
        DebugLog.add("Searching at $latitude, $longitude")
        // Clear the arrays
        //relations.clear()
        //artworks.clear()
        val tmpArtworks = mutableListOf<Artwork>()
        val tmpRelations = mutableListOf<Relation>()
        val searchTerm = if (showMuseum) "\"tourism\"~\"artwork|museum\"" else "\"tourism\"=\"artwork\""
        val query = """
        [out:json];
        (
        node[$searchTerm](around:$radius,$latitude,$longitude);
        way[$searchTerm](around:$radius,$latitude,$longitude);
        relation["type"="site"](around:$radius,$latitude,$longitude);
        );
        out center;
        """.trimIndent()

        val url = "$selectedPoiServer?data=${query.replace("\n", "")}"
        val client = sharedOkHttpClient //OkHttpClient()
        val request = Request.Builder().url(url).build()
        var success = false
        var elapsedTime: String
        var response: Response
        try {
            val elapsedNanos = measureNanoTime { response = client.newCall(request).execute() }
            elapsedTime = nanoToTime(elapsedNanos)
        } catch (e: IOException) {
            DebugLog.add("$e")
            withContext(Dispatchers.Main) { Toast.makeText(context, context.getString(R.string.error) + ": " + e.message, Toast.LENGTH_LONG).show() }
            return@withContext artworksGeoJson // Don't return null, rather keep whatever is in the array
        }
        // Website errors
        if (response.code != 200) {
            val message = context.getString(R.string.error) + ": " + getHttpErrorMessage(response.code)
            DebugLog.add(message)
            withContext(Dispatchers.Main) { Toast.makeText(context, message, Toast.LENGTH_LONG).show() }
            return@withContext artworksGeoJson // Don't return null, rather keep whatever is in the array
        }

        // Call successful but no JSON data?
        val responseBody = response.body.string()
        if (responseBody.isBlank()) {
            val message = context.getString(R.string.error) + ": " + getHttpErrorMessage(response.code)
            DebugLog.add(message)
            withContext(Dispatchers.Main) { Toast.makeText(context, message, Toast.LENGTH_LONG).show() }
            return@withContext artworksGeoJson // Don't return null, rather keep whatever is in the array
        }

        try {
            val json = JSONObject(responseBody)
            val elements = json.getJSONArray("elements")
            for (i in 0 until elements.length()) {
                val element = elements.getJSONObject(i)
                val type = element.optString("type")
                val id = element.optString("id")
                if (type.isBlank() || id.isBlank()) { // OSM should prevent this, but well…
                    DebugLog.add("Malformed node 'id' or 'type'")
                    continue
                }
                // Coordinates
                when (type) {
                    "node" -> {
                        // Nodes have lat/lon directly
                        val latValue = element.optDouble("lat", Double.NaN)
                        val lonValue = element.optDouble("lon", Double.NaN)
                        val latLonPair =
                            if (latValue.isNaN() || lonValue.isNaN()) null else Pair(
                                latValue,
                                lonValue
                            )
                        if (latLonPair == null) continue // Skip if no valid coordinates

                        var nodeType = NodeType.UNKNOWN
                        val tags = element.optJSONObject("tags")
                        val keyValueStore = mutableMapOf<String, String>()
                        if (tags != null) {
                            val keys = tags.keys()
                            while (keys.hasNext()) {
                                val key = keys.next()
                                val value = tags.optString(key)
                                keyValueStore[key] = value
                            }

                            nodeType = getNodeType(tags)
                        }
                        tmpArtworks.add(
                            Artwork(
                                id = id,
                                lat = latLonPair.first,
                                lon = latLonPair.second,
                                tags = keyValueStore,
                                type = nodeType
                            )
                        )
                    }

                    "way" -> {
                        // Nodes have lat/lon directly
                        val center = element.getJSONObject("center")
                        val latValue = center.optDouble("lat", Double.NaN)
                        val lonValue = center.optDouble("lon", Double.NaN)
                        val latLonPair =
                            if (latValue.isNaN() || lonValue.isNaN()) null else Pair(
                                latValue,
                                lonValue
                            )
                        if (latLonPair == null) continue // Skip if no valid coordinates

                        var nodeType = NodeType.UNKNOWN
                        val tags = element.optJSONObject("tags")
                        val keyValueStore = mutableMapOf<String, String>()
                        if (tags != null) {
                            val keys = tags.keys()
                            while (keys.hasNext()) {
                                val key = keys.next()
                                val value = tags.optString(key)
                                keyValueStore[key] = value
                            }
                            nodeType = getNodeType(tags)
                        }
                        tmpArtworks.add(
                            Artwork(
                                id = id,
                                lat = latLonPair.first,
                                lon = latLonPair.second,
                                tags = keyValueStore,
                                type = nodeType
                            )
                        )
                    }

                    "relation" -> {
                        // Capture tags
                        val tags = element.optJSONObject("tags")
                        val tagsKeyValueStore = mutableMapOf<String, String>()
                        if (tags != null) {
                            val keys = tags.keys()
                            while (keys.hasNext()) {
                                val key = keys.next()
                                val value = tags.optString(key)
                                tagsKeyValueStore[key] = value
                            }
                        }
                        // Capture members
                        val memberArray: ArrayList<String> = arrayListOf()
                        val members = element.optJSONArray("members")
                        if (members != null) {
                            for (i in 0 until members.length()) {
                                val member = members.getJSONObject(i)
                                val type = member.optString("type")
                                val role = member.optString("role")
                                if (type == "node" && role == "member") {
                                    val ref = member.optLong("ref")
                                    memberArray.add(ref.toString())
                                }
                            }
                        }
                        tmpRelations.add(
                            Relation(
                                id = id,
                                tags = tagsKeyValueStore,
                                members = memberArray
                            )
                        )
                    }
                }
            }
            success = true
        } catch (e: Exception ){
            withContext(Dispatchers.Main) { Toast.makeText(context, context.getString(R.string.error) + ": " + e.message, Toast.LENGTH_LONG).show() }
            DebugLog.add("Exception: $e")
        }

        if (success) {
            if (tmpArtworks.isNotEmpty()) {
                // Overwrite only if we were able to successfully connect
                artworks = tmpArtworks
                relations = tmpRelations
                artworksGeoJson = overpassToGeoJson()
                artworkCount.value = artworks.size
            }
            withContext(Dispatchers.Main) {
                if (artworks.isEmpty()) { Toast.makeText(context, context.getString(R.string.noArtworksInCircle, radius), Toast.LENGTH_LONG).show() }
                else {
                    val msg = context.getString(R.string.nrArtworksInCircle, tmpArtworks.size) + if (expertMode) " ($elapsedTime)" else ""
                    Toast.makeText(context, msg, Toast.LENGTH_LONG).show()
                }
            }
            return@withContext artworksGeoJson
        }
        return@withContext null
    }

    // Launch OSM update in the context of the viewModel so it survives AlertDialog closure
    fun updateOsm(selectedArtwork: Artwork, token: String) {
        if (sandboxMode) return
        if (token.isEmpty()) return
        val snapshot = selectedArtwork.copy(
            tags = selectedArtwork.tags.toMap() as MutableMap<String, String> // <- convert SnapshotStateMap to regular Map
        )
        viewModelScope.launch(Dispatchers.IO) {
            try {
                // Call your existing suspend function
                val msg = createDataset(snapshot,application, token)
                withContext(Dispatchers.Main) { Toast.makeText(application, msg, Toast.LENGTH_LONG).show() }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) { Toast.makeText(application, application.getString(R.string.updateFailed) + ": " + e.message, Toast.LENGTH_LONG).show() }
            }
        }
    }

    // Helper
    fun getNodeType(tags: JSONObject): Int {
        val tourism = try { (tags["tourism"] ?: "").toString() } catch (_: Exception) { "" }
        val artworkType = try { (tags["artwork_type"] ?: "").toString() } catch (_: Exception) { "" }

        return when {
            tourism.contains("museum", ignoreCase = true) -> NodeType.MUSEUM
            artworkType.contains("statue", ignoreCase = true) -> NodeType.STATUE
            artworkType.contains("sculpture", ignoreCase = true) -> NodeType.SCULPTURE
            artworkType.contains("graffiti", ignoreCase = true) -> NodeType.GRAFFITI
            artworkType.contains("mural", ignoreCase = true) -> NodeType.MURAL
            artworkType.contains("relief", ignoreCase = true) -> NodeType.RELIEF
            artworkType.contains("installation", ignoreCase = true) -> NodeType.INSTALLATION
            artworkType.contains("bust", ignoreCase = true) -> NodeType.BUST
            artworkType.contains("architecture", ignoreCase = true) -> NodeType.ARCHITECTURE
            artworkType.contains("stone", ignoreCase = true) -> NodeType.STONE
            artworkType.contains("fountain", ignoreCase = true) -> NodeType.FOUNTAIN
            else -> NodeType.UNKNOWN
        }
    }

    // Parse artwork array and create a proper geoJson string
    fun overpassToGeoJson(): JSONObject {
        val features = JSONArray()
        var i = 0
        for (item in artworks) { //newArtworks) {
            val feature = JSONObject()
            feature.put("type", "Feature")
            feature.put("geometry", JSONObject().apply {
                put("type", "Point")
                put("coordinates", JSONArray(listOf(item.lon, item.lat)))
            })
            feature.put("properties", JSONObject().apply {
                put("array_id", i.toString())
                put("artwork_type", item.type.toString())
            })
            i++

            features.put(feature)
        }
        val featureCollection = JSONObject()
        featureCollection.put("type", "FeatureCollection")
        featureCollection.put("features", features)
        return featureCollection
    }

    sealed class UploadState {
        object Idle : UploadState()
        object Uploading : UploadState()
        data class Success(val filename: String) : UploadState()
        data class Error(val message: String) : UploadState()
    }

    // Simplified mutable state
    var uploadState by mutableStateOf<UploadState>(UploadState.Idle)
        private set

    fun uploadWikidataImage(context: Context, accessToken: String, uri: Uri?, categories: List<String>) {
        viewModelScope.launch {
            uploadState = UploadState.Uploading

            try {
                val csrfToken = withContext(Dispatchers.IO) { getCsrfToken(accessToken) }
                if (csrfToken == null) {
                    uploadState = UploadState.Error("Failed to get upload token")
                    return@launch
                }

                val material = if (currentTags["material"].isNullOrEmpty()) "" else currentTags["material"] + " "
                val name = if (currentTags["name"].isNullOrEmpty()) "with no or unknown name" else "named " + currentTags["name"]
                val artist = "by " +if (currentTags["artist_name"].isNullOrEmpty()) "unknown" else currentTags["artist_name"]
                val artworkType = if (currentTags["artwork_type"].isNullOrEmpty()) "artwork" else currentTags["artwork_type"]
                val description = "This is a photo of the ${artworkType}."
                val caption = "$material$artworkType $artist $name".replaceFirstChar { it.titlecase() }

                val filename = withContext(Dispatchers.IO) {
                    uploadImageToCommons(context, accessToken, csrfToken, uri, categories, description, caption)
                }

                if (filename != null) {
                    uploadState = UploadState.Success(filename)
                } else {
                    uploadState = UploadState.Error("Upload failed")
                }

            } catch (e: Exception) {
                uploadState = UploadState.Error("Upload error: ${e.message}")
                Log.e("WikiUpload", "Upload exception", e)
            }
        }
    }
}
