package com.zell_mbc.publicartexplorer

import androidx.annotation.DrawableRes
import androidx.annotation.StringRes

// Preferences keys
const val SUBSCRIPTION_ACTIVE_KEY = "unlock_expert_mode" // Dual use: Name of the product in Google Play, preferences key
const val EXPERT_MODE_KEY = "expert_mode"
const val DEBUG_MODE_KEY = "debug_mode"
const val RADIUS_KEY = "radius"
const val LONGITUDE_KEY = "longitude"
const val LATITUDE_KEY = "latitude"
const val ZOOM_KEY = "zoom"
const val SHOW_MUSEUM_KEY = "show_museum"
const val LAUNCH_AT_CURRENT_POSITION_KEY = "launch_at_current_position"
const val MAP_STYLE_KEY = "map_style"
const val POI_SERVER_KEY = "poi_server"
const val TILE_SERVER_KEY = "tile_server"
const val ENABLE_HAPTIC_FEEDBACK_KEY = "enable_haptic_feedback"
const val MAPTOOLKIT_API_KEY = "maptoolkit_api_key" // Used for FOSS flavour where user (I) might have his own key
const val MAPBOX_API_KEY = "mapbox_api_key" // Used for FOSS flavour where user (I) might have his own key
const val DARK_MODE_KEY = "dark_mode"

// Minimum distance to move the camera (in meters)
const val MIN_MOVEMENT_METERS = 1.0

enum class DarkThemeOptions {
    Light, Dark, FollowSystem
}
// Preferences
object Constants {
    const val DEFAULT_ZOOM = 14.0 // 0,5km
    const val DEFAULT_EXPERT_MODE = false
    const val DEFAULT_SHOW_MUSEUM = false
    const val DEFAULT_LAUNCH_AT_CURRENT_POSITION = false
    const val DEFAULT_ZOOM_LEVEL_FILTERING = false

    const val DEFAULT_RADIUS = 5000
}

object NodeType {
    const val UNKNOWN = 0
    const val SCULPTURE = 1
    const val STATUE = 2
    const val MUSEUM = 3
    const val GRAFFITI = 4
    const val MURAL = 5
    const val RELIEF = 6
    const val INSTALLATION = 7
    const val FOUNTAIN = 8
    const val ARCHITECTURE = 9
    const val BUST = 10
    const val STONE = 11
}

val httpErrorMessages = mapOf(
    400 to "Bad Request",
    401 to "Unauthorized",
    403 to "Forbidden",
    404 to "Not Found",
    408 to "Request Timeout",
    429 to "Too Many Requests",
    500 to "Internal Server Error",
    502 to "Bad Gateway",
    503 to "Service Unavailable",
    504 to "Gateway Timeout"
)
fun getHttpErrorMessage(code: Int): String = httpErrorMessages[code] ?: "HTTPS $code"

// POI related constants
data class OsmServer(val name: String, val url: String)
val poiServers = listOf(
    OsmServer("Main OSM Server", "https://overpass-api.de/api/interpreter"),
    OsmServer("French Mirror", "https://overpass.openstreetmap.fr/api/interpreter"),
    OsmServer("QGIS Mirror (Switzerland)", "https://lz4.overpass-api.de/api/interpreter"),
    OsmServer("PublicArtExplorer Server (Experimental)", "https://overpass.zell-mbc.com/api/interpreter")
)
const val DEFAULT_POI_SERVER = 0

// Tile related constants
val openFreeMapStyles = listOf("Bright","Liberty","Positron","Fiord","Dark")
val mapToolKitStyles = listOf("City", "Terrain","Terrainwinter", "Light", "Green", "Dark")
const val DEFAULT_MAPTOOLKIT_STYLE = 4
const val DEFAULT_OPENFREEMAP_STYLE = 2

val tileServers = listOf(
    OsmServer("OpenFreeMap", "https://tiles.openfreemap.org/styles/"),
    OsmServer("Mapbox (Commercial)", "mapbox_styles"),
    OsmServer("Maptoolkit (Commercial)", "https://static.maptoolkit.net/rapidapi/styles/")
)
const val OPENFREEMAP_TILE_SERVER = 0
const val MAPBOX_TILE_SERVER  = 1
const val MAPTOOLKIT_TILE_SERVER  = 2

// OSM stuff
const val OSM_DEBUG = false
val osmAppId   = if (OSM_DEBUG) "d8lgVwJ4sJ4V0xKi8OSDzqtbXZeb1slfnnqhRkRuEcg" else "Nmr_1DAQFJJjuAxSGOSq2iyEPcjWQldmMoBYfsp4G_4"
val osmApiUrl  = if (OSM_DEBUG) "https://master.apis.dev.openstreetmap.org" else "https://api.openstreetmap.org"
val osmAuthUrl = if (OSM_DEBUG) "https://master.apis.dev.openstreetmap.org/oauth2" else "https://www.openstreetmap.org/oauth2"

const val oAuthRedirectUri = "https://oauth2.publicartexplorer.com/osm-callback"
const val osmScope = "read_prefs write_api"

val wikiAppId = "101ac98751b45657e2d4626a09e153fd"  //"5bf678d7550d10167ac432493c921fde"
val wikimediaAuthEndpoint = "https://meta.wikimedia.org/w/rest.php/oauth2/authorize"
val wikimediaTokenEndpoint = "https://meta.wikimedia.org/w/rest.php/oauth2/access_token"
const val wikiScope = "" //"read upload writeapi"

val OSM_ACCESS_TOKEN_KEY = if (OSM_DEBUG) "access_token_debug" else "access_token"
const val WIKI_ACCESS_TOKEN_KEY = "wiki_access_token"
const val WIKI_REFRESH_TOKEN_KEY = "wiki_refresh_token"

// Data class for artwork
data class Artwork(
    val id: String,
    val lat: Double,
    val lon: Double,
    var tags: MutableMap<String, String> = mutableMapOf(),
    val type: Int
)

const val maptoolkitName = "Maptoolkit"

// Data class for artwork relations
data class Relation(
    val id: String,
    val members: ArrayList<String> = arrayListOf(),
    val tags: MutableMap<String, String> = mutableMapOf()
)

object ProjectDetails {
    const val AUTHOR = "Zell-MBC"
    const val AUTHOR_URL = "https://zell-mbc.com"
    const val WEBSITE_URL = "https://PublicArtExplorer.com"
    const val EMAIL_ADDRESS = "publicartexplorer@zell-mbc.com"
    const val FEDIVERSE_URL = "https://social.zell-mbc.com/publicartexplorer"
    const val THIRD_PARTY = "https://codeberg.org/toz/PublicArtExplorer"
    const val APP_LICENSE = "https://codeberg.org/toz/PublicArtExplorer/src/branch/main/LICENSE"
    const val EXPERT_MODE_URL = "https://PublicArtExplorer.com" // Do not delete, used in Play flavour
}

// Navigation
sealed class Screen(val route: String, @param:StringRes val label: Int, @param:DrawableRes val icon: Int) {
    object Map : Screen("map", R.string.tab_map, R.drawable.outline_map_24)
    object Settings : Screen("settings", R.string.tab_settings, R.drawable.outline_settings_24)
    object Info : Screen("info", R.string.tab_info, R.drawable.outline_info_24)
}

val bottomNavItems = listOf(
    Screen.Map,
    Screen.Settings,
    Screen.Info
)
