package com.zell_mbc.publicartexplorer.maps

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.location.Location
import android.location.LocationListener
import android.location.LocationManager
import android.location.LocationRequest
import android.os.Build
import android.os.Bundle
import android.os.CancellationSignal
import androidx.core.app.ActivityCompat
import java.util.concurrent.Executors


// Request location update, execute callback
fun getLocation(context: Context, callback: (Location?) -> Unit) {
    if (ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED &&
        ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED ) {
        callback(null)
        return
    }

    val locationManager = context.getSystemService(Context.LOCATION_SERVICE) as LocationManager

    // Check if providers are enabled
    val isGpsEnabled = locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)
    val isNetworkEnabled = locationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER)

    // No provider enabled? Get the hell out of here
    if (!isGpsEnabled && !isNetworkEnabled) {
        callback(null)
        return
    }

    val provider = if (isGpsEnabled) LocationManager.GPS_PROVIDER else LocationManager.NETWORK_PROVIDER

    var callbackInvoked = false
    val executor = Executors.newSingleThreadExecutor()

    // Manual timeout safeguard (e.g. 10 seconds)
    executor.execute {
        Thread.sleep(10_000)
        if (!callbackInvoked) {
            callbackInvoked = true
            callback(null)
        }
    }

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
        val locationRequest = LocationRequest.Builder(1000).setQuality(LocationRequest.QUALITY_HIGH_ACCURACY).build()
        locationManager.getCurrentLocation(provider, locationRequest, CancellationSignal(), Executors.newSingleThreadExecutor()) { location ->
            // This is the callback for current SDKs!
            locationManager.getCurrentLocation(provider, locationRequest, CancellationSignal(), executor) { location ->
                if (!callbackInvoked) {
                    callbackInvoked = true
                    callback(location)
                }
            }
        }
    } else {
        // For older APIs, create a full LocationListener implementation
        val listener = object : LocationListener {
            override fun onLocationChanged(location: Location) {
                if (!callbackInvoked) {
                    callbackInvoked = true
                    callback(location)
                }
                locationManager.removeUpdates(this)
            }

            override fun onStatusChanged(provider: String?, status: Int, extras: Bundle?) {}
            override fun onProviderEnabled(provider: String) {}
            override fun onProviderDisabled(provider: String) {}
        }

        locationManager.requestSingleUpdate(provider, listener, null)
    }
}
