/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.billing

import android.app.Activity
import android.content.Context
import android.content.SharedPreferences
import android.util.Log
import androidx.preference.PreferenceManager
import com.android.billingclient.api.AcknowledgePurchaseParams
import com.android.billingclient.api.BillingClient
import com.android.billingclient.api.BillingClientStateListener
import com.android.billingclient.api.BillingFlowParams
import com.android.billingclient.api.BillingResult
import com.android.billingclient.api.PendingPurchasesParams
import com.android.billingclient.api.QueryProductDetailsParams
import com.android.billingclient.api.QueryPurchasesParams
import androidx.core.content.edit
import com.android.billingclient.api.Purchase
import com.zell_mbc.medilog.UnlimitedRecords


class BillingManagerImpl() : BillingManager {
    private var billingClient: BillingClient? = null
    private var isBillingReady = false
    private var pendingCheck: (() -> Unit)? = null
    private lateinit var preferences: SharedPreferences
    private var purchasingProcessActive = false

    override var showThankYouDialog: ((Boolean) -> Unit)? = null
    override var setUnlimitedRecords: ((Boolean) -> Unit)? = null
    override var showBillingUnavailableDialog: ((Boolean) -> Unit)? = null

    private fun getBillingClient(context: Context): BillingClient {
        if (billingClient == null) {
            billingClient = BillingClient.newBuilder(context)
                .setListener { billingResult, purchases ->
                    preferences = PreferenceManager.getDefaultSharedPreferences(context)
                    when {
                    billingResult.responseCode == BillingClient.BillingResponseCode.OK && purchases != null -> {
                        for (purchase in purchases) {
                            if (purchase.products.contains(UnlimitedRecords)) {
                                // Unlock premium features
                                preferences.edit { putBoolean(UnlimitedRecords, true) }

                                // Acknowledge purchase if not yet acknowledged
                                if (purchase.purchaseState == Purchase.PurchaseState.PURCHASED && !purchase.isAcknowledged) {
                                    val acknowledgeParams = AcknowledgePurchaseParams.newBuilder().setPurchaseToken(purchase.purchaseToken).build()
                                    billingClient?.acknowledgePurchase(acknowledgeParams) { ackResult ->
                                        if (ackResult.responseCode == BillingClient.BillingResponseCode.OK) {
                                            preferences.edit { putBoolean(UnlimitedRecords, true) }
                                            Log.d("Billing", "Purchase acknowledged")
                                            showThankYouDialog?.invoke(true)
                                            setUnlimitedRecords?.invoke(true)
                                        } else {
                                            val message = "Error acknowledging purchase: ${ackResult.debugMessage}"
                                            Log.w("Billing", message)
                                            //Handler(Looper.getMainLooper()).post { Toast.makeText(context, message, Toast.LENGTH_LONG).show() }
                                        }
                                    }
                                }
                            }
                        }
                    }
                    billingResult.responseCode == BillingClient.BillingResponseCode.ITEM_ALREADY_OWNED -> {
                        Log.w("Billing", "Item already owned")
                        preferences.edit { putBoolean(UnlimitedRecords, true) }
                        setUnlimitedRecords?.invoke(true)
                        }
                    billingResult.responseCode == BillingClient.BillingResponseCode.USER_CANCELED -> {
                        Log.w("Billing", "User canceled")
                        preferences.edit { putBoolean(UnlimitedRecords, false) }
                    }
                    else ->{
                        Log.w("Billing", "Purchase failed: ${billingResult}")
                        preferences.edit { putBoolean(UnlimitedRecords, false) }
                        }
                    }
                }
                .enablePendingPurchases(PendingPurchasesParams.newBuilder().enableOneTimeProducts().build())
                .enableAutoServiceReconnection()
                .build()

            billingClient?.startConnection(object : BillingClientStateListener {
                override fun onBillingSetupFinished(billingResult: BillingResult) {
                    if (billingResult.responseCode == BillingClient.BillingResponseCode.OK) {
                        Log.d("Billing", "Billing setup finished")
                        isBillingReady = true
                        pendingCheck?.invoke()
                        pendingCheck = null
                    } else {
                        // Billing is not available: show appropriate message to user
                        Log.w("Billing", "Billing unavailable: ${billingResult.responseCode} (${billingResult.debugMessage})")
                        if (purchasingProcessActive) {
                            showBillingUnavailableDialog?.invoke(true)
                            purchasingProcessActive = false
                        } // Ignore otherwise
                    }
                }

                override fun onBillingServiceDisconnected() {
                    Log.d("Billing", "Billing service disconnected")
                    isBillingReady = false
                }
            })
        }

        return billingClient!!
    }


    override fun isLicensedUser(context: Context) {
        var unlimitedRecords = false
        val billingClient = getBillingClient(context)

        if (!isBillingReady) {
            // Defer the check until ready
            pendingCheck = { isLicensedUser(context) }
            Log.d("Billing", "Not ready")
            return
        }
        billingClient.queryPurchasesAsync(
            QueryPurchasesParams.newBuilder()
                .setProductType(BillingClient.ProductType.INAPP)
                .build()
        ) { billingResult, purchasesList ->
            unlimitedRecords = purchasesList.any { it.products.contains(UnlimitedRecords) }
            preferences = PreferenceManager.getDefaultSharedPreferences(context)
            preferences.edit { putBoolean(UnlimitedRecords, unlimitedRecords) } // Can be true or false!
            setUnlimitedRecords?.invoke(unlimitedRecords)
            Log.d("Billing", billingResult.toString() + ", UnlimitedRecords=$unlimitedRecords")
        }
        return
    }

    // Process purchase
    override fun launchPurchaseFlow(activity: Activity) {
        val queryProductDetailsParams = QueryProductDetailsParams.newBuilder()
            .setProductList(
                listOf(
                    QueryProductDetailsParams.Product.newBuilder()
                        .setProductId(UnlimitedRecords) // your product ID here
                        .setProductType(BillingClient.ProductType.INAPP)
                        .build()
                )
            )
            .build()

        billingClient?.queryProductDetailsAsync(queryProductDetailsParams) { billingResult, queryProductDetailsResult ->
            if (billingResult.responseCode == BillingClient.BillingResponseCode.OK) {
                val productDetailsList = queryProductDetailsResult.productDetailsList
                if (productDetailsList.isNotEmpty()) {
                    val productDetails = productDetailsList[0]
                    // Proceed with purchase flow
                    val billingFlowParams = BillingFlowParams.newBuilder()
                        .setProductDetailsParamsList(
                            listOf(
                                BillingFlowParams.ProductDetailsParams.newBuilder()
                                    .setProductDetails(productDetails)
                                    .build()
                            )
                        )
                        .build()

                    // This will trigger the billingClient.setListener defined above
                    billingClient?.launchBillingFlow(activity, billingFlowParams)
                }
                // Optionally handle unfetched products:
                val unfetchedList = queryProductDetailsResult.unfetchedProductList
                if (unfetchedList.isNotEmpty()) {
                    Log.w("Billing", "Some products could not be fetched: $unfetchedList")
                    // Optionally notify the user or fallback gracefully
                    //showBillingErrorToUser("Unable to fetch all products. Try again later.")
                }
            }
        }
    }

    // Check if unlimitedRecords product is available, if so, return product details
    override fun queryUpgradeProduct(context: Context, onResult: (UpgradeProductInfo?) -> Unit) {
        val params = QueryProductDetailsParams.newBuilder()
            .setProductList(
                listOf(
                    QueryProductDetailsParams.Product.newBuilder()
                        .setProductId(UnlimitedRecords)
                        .setProductType(BillingClient.ProductType.INAPP)
                        .build()
                )
            )
            .build()
        purchasingProcessActive = true
        billingClient?.queryProductDetailsAsync(params) { billingResult, result ->
            val details = result.productDetailsList.firstOrNull()
            if (details != null) {
                onResult(
                    UpgradeProductInfo(
                        title = details.title,
                        description = details.description,
                        price = details.oneTimePurchaseOfferDetails?.formattedPrice ?: ""
                    )
                )
            } else {
                onResult(null)
            }
        }
    }
}
