/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.support

import android.content.Context
import android.content.SharedPreferences
import android.content.pm.PackageManager
import android.os.Build
import android.widget.Toast
import androidx.annotation.RequiresApi
import androidx.core.content.edit
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.ACTIVE_TABS_KEY
import com.zell_mbc.medilog.INITIAL_TABS
import com.zell_mbc.medilog.MainActivity.Companion.Delimiter.THRESHOLD
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.Tabs.BLOODPRESSURE
import com.zell_mbc.medilog.Tabs.DIARY
import com.zell_mbc.medilog.Tabs.DOCUMENTS
import com.zell_mbc.medilog.Tabs.FLUID
import com.zell_mbc.medilog.Tabs.GLUCOSE
import com.zell_mbc.medilog.Tabs.OXIMETRY
import com.zell_mbc.medilog.Tabs.TEMPERATURE
import com.zell_mbc.medilog.Tabs.WEIGHT
import com.zell_mbc.medilog.data.DataViewModel
import com.zell_mbc.medilog.preferences.SettingsActivity
import java.io.ByteArrayInputStream
import java.io.InputStream
import java.security.cert.CertificateException
import java.security.cert.CertificateFactory
import java.security.cert.X509Certificate
import java.text.DateFormat
import java.text.DecimalFormatSymbols
import java.time.Instant
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.ZoneId
import java.time.ZoneOffset
import java.time.ZonedDateTime
import java.util.Locale
import kotlin.text.replace

// Return a map of tab id and tab name
fun getAllTabMeta(context: Context): Map<Int, String> {
    val preferences = PreferenceManager.getDefaultSharedPreferences(context)

    // Check if a custom name was set
    fun getLabel(prefKey: String, defaultResId: Int): String {
        val custom = preferences.getString(prefKey, "")?.trim().orEmpty()
        return if (custom.isNotEmpty()) custom else context.getString(defaultResId)
    }

    return mapOf(
        WEIGHT to getLabel(SettingsActivity.KEY_PREF_WEIGHT_CUSTOM_NAME, R.string.weight),
        BLOODPRESSURE to context.getString(R.string.bloodPressure),
        DIARY to context.getString(R.string.diary),
        FLUID to getLabel(SettingsActivity.KEY_PREF_FLUID_CUSTOM_NAME, R.string.fluid),
        TEMPERATURE to getLabel(SettingsActivity.KEY_PREF_TEMPERATURE_CUSTOM_NAME, R.string.temperature),
        OXIMETRY to context.getString(R.string.oximetry),
        GLUCOSE to getLabel(SettingsActivity.KEY_PREF_GLUCOSE_CUSTOM_NAME, R.string.glucose),
        DOCUMENTS to context.getString(R.string.documents)
    )
}

fun loadEnabledTabs(preferences: SharedPreferences?): List<Int> {
    val raw = preferences?.getString(ACTIVE_TABS_KEY, INITIAL_TABS) ?: INITIAL_TABS
    return raw.split(',')
        .mapNotNull { it.toIntOrNull() }  // safely convert to Int
        .toList()
}

fun checkMinMax(field: String, min: Int, max: Int): Boolean =
    checkMinMax(field, min.toFloat(), max.toFloat())

fun checkMinMax(field: String, min: Float, max: Float): Boolean {
    if (field.isEmpty()) return true
    val numericValue = field.replace(',', '.').toFloatOrNull() ?: return false
    return numericValue in min..max
}

// Matches input to template and decides if cursor should move to new field or not
fun shouldMoveFocus(field: DataViewModel.EditField): Boolean {
    return shouldMoveFocus(field.value, field.template)
}

// Matches input to template and decides if cursor should move to new field or not
fun shouldMoveFocus(newValue: String, template: String, maxValue: Int? = null): Boolean {
    val normalizedValue = newValue.replace(',', '.')
    val normalizedTemplate = template.replace(',', '.')
    val separatorIndex = normalizedTemplate.indexOf('.')

    val maxBefore = if (separatorIndex >= 0) separatorIndex else template.length
    val maxAfter = if (separatorIndex >= 0) template.length - separatorIndex - 1 else 0

    val parts = normalizedValue.split('.')

    val before = parts.getOrNull(0)?.length ?: 0
    val after = parts.getOrNull(1)?.length ?: 0

    if (maxAfter == 0 && before > 0 && maxValue != null) {
        val intPart = parts.getOrNull(0) ?: ""
        val intValue = intPart.toIntOrNull()

        if (intValue != null) {
            val maxDigits = maxValue.toString().length

            if (before >= maxDigits)
                return true

            val maxFirstDigit = maxValue.toString().firstOrNull()?.digitToIntOrNull() ?: 0
            val inputFirstDigit = intPart.firstOrNull()?.digitToIntOrNull() ?: 0

            // Jump if value starts with digit > max
            if (maxDigits > 1 && inputFirstDigit >= maxFirstDigit) {
                if (before >= maxDigits - 1)
                    return true
            }
        }
    }

    if (before == maxBefore && maxAfter == 0 && !normalizedValue.endsWith("."))
        return true

    if (after == maxAfter && maxAfter > 0)
        return true

    if (before > maxBefore || after > maxAfter)
        return true

    return false
}



fun matchesTemplate(field: DataViewModel.EditField): Boolean {
    return matchesTemplate(field.value, field.template)
}

fun matchesTemplate(input: String, template: String): Boolean {
    val normalizedInput = input.replace(',', '.')
    var normalizedTemplate = normalizeDecimalSeparator(template)

    // Should never happen… but just in case: replace everything except '.' with '0'
    normalizedTemplate = normalizedTemplate.map { c -> if (c == '.') '.' else '0' }.joinToString("")

    // First, validate the template format
    val templateParts = normalizedTemplate.split('.')
    val decimalPos = normalizedTemplate.indexOf('.')
    if (templateParts.size > 2 || templateParts.any { it.any { c -> c != '0' } }) {
        throw IllegalArgumentException("Invalid template format: only '0' allowed with optional single dot, template: $template")
    }

    // Determine max lengths
    val maxIntDigits = templateParts[0].length
    val maxFracDigits = if (templateParts.size == 2) templateParts[1].length else 0

    // Build regex pattern
    val regex = if (decimalPos >= 0) {
        // Accept: full match, partial input, or just decimal
        Regex("""^\d{0,${maxIntDigits}}(\.\d{0,${maxFracDigits}})?$""")
    } else {
        Regex("""^\d{0,${maxIntDigits}}$""")
    }

    return regex.matches(normalizedInput)
}

fun normalizeDecimalSeparator(input: String): String {
    return input.replace(',', '.')
}

// Converts value to string, formated along it's template
fun valueToString(value: String, template: String): String {
    if (value.isEmpty()) return "" // Do nothing

    val normalizedValue = value.replace(',', '.') // Just in case
    val normalizedTemplate = template.replace(',', '.')

    // Determine if the template expects a float (by presence of '.')
    val isFloat = normalizedTemplate.contains('.')
    val fractionDigits = if (isFloat) {
        val index = normalizedTemplate.indexOf('.')
        normalizedTemplate.length - index - 1
    } else 0

    val valueAsString = try {
        if (isFloat) {
            val number = normalizedValue.toFloat()
            //       highlightRow = (highlightValues && ((number < lowerThreshold) || (number > upperThreshold)))
            "%.${fractionDigits}f".format(number)
        } else {
            val number = normalizedValue.toInt()
            //      highlightRow = (highlightValues && ((number < lowerThreshold) || (number > upperThreshold)))
            number.toString()
        }
    } catch (_: NumberFormatException) {
        //if (isFloat) "%.${fractionDigits}f".format(0f) else "0"
        value
    }

    return valueAsString
}

fun currentLocale(context: Context): String {
    val currentLocale = context.resources.configuration.locales.get(0)
    return currentLocale.language // e.g., "en", "fr", "de"
}

fun getDecimalSeparator(locale: Locale = Locale.getDefault()): String {
    return DecimalFormatSymbols.getInstance(locale).decimalSeparator.toString()
}

fun checkThresholds(context: Context, value: String, default: String, type: String): Array<String> {
    var lowerThreshold = 0f
    var upperThreshold = 99999f

    // Check if value is empty, if yes set to default
    var checkValue = value.ifEmpty { default }
    checkValue = checkValue.replace(',', '.') // normalize decimal

    // Is only the lower limit supplied, if yes add fake upper threshold
    if (checkValue[checkValue.length-1].toString() == THRESHOLD)
        checkValue += upperThreshold.toString()

    // Is only the upper limit supplied, if yes add fake lower threshold
    if (checkValue[0].toString() == THRESHOLD)
        checkValue = lowerThreshold.toString() + checkValue

    if (checkValue.isNotEmpty()) {
        val minMax = checkValue.split(THRESHOLD)
        // Check if really 2 values were present
        if ( minMax.size == 2 ) {
            try {
                lowerThreshold = minMax[0].toFloat()
            } catch (_: NumberFormatException) {
                Toast.makeText(context, type + ", " + context.getString(R.string.invalidLowerThreshold) + ": $checkValue", Toast.LENGTH_LONG).show()
                checkValue = default
            }
            try {
                upperThreshold = minMax[1].toFloat()
            } catch (_: NumberFormatException) {
                Toast.makeText(context,type + ", " + context.getString(R.string.invalidUpperThreshold) + ": $checkValue", Toast.LENGTH_LONG).show()
                //userOutputService.showAndHideMessageForLong(context.getString(type) + ", " + context.getString(R.string.invalidUpperThreshold) + ": $checkValue")
                checkValue = default
            }

            if (upperThreshold <= lowerThreshold) {
                Toast.makeText(context,type + ", " + context.getString(R.string.invalidThresholds) + ": $checkValue", Toast.LENGTH_LONG).show()
                //userOutputService.showAndHideMessageForLong(context.getString(type) + ", " + context.getString(R.string.invalidThresholds) + ": $checkValue")
                checkValue = default
            }
        }
        else {
            Toast.makeText(context,type + ", " + context.getString(R.string.invalidThresholds) + ": $checkValue", Toast.LENGTH_LONG).show()
            //userOutputService.showAndHideMessageForLong(context.getString(type) + ", " + context.getString(R.string.invalidThresholds) + ": $checkValue")
            checkValue = default
        }
    }
    else checkValue = default

    val finalMinMax = checkValue.split(THRESHOLD)
    val arr = try {
        arrayOf(finalMinMax[0], finalMinMax[1])
    } catch (_: IndexOutOfBoundsException) {
        // This exception can only happen if the default value is wrong -> Should never happen…
        Toast.makeText(context,type + ", " + context.getString(R.string.invalidThresholds) + ": $checkValue", Toast.LENGTH_LONG).show()
        //userOutputService.showAndHideMessageForLong(context.getString(type) + ", " + context.getString(R.string.invalidThresholds) + ": $checkValue")
        arrayOf("0","0")
    }
    return arr
}

fun getMemoryConsumption(): Long {
    val usedSize = try {
        val info = Runtime.getRuntime()
        val freeSize = info.freeMemory()
        val totalSize = info.totalMemory()
        totalSize - freeSize
    } catch (_: Exception) {
        -1L
    }
    return usedSize
}

fun getCorrectedDateFormat(c: Context): DateFormat {
    // For some reason en_DE does not set the correct date format?
//    val devLocale = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) c.resources.configuration.locales.get(0) else c.resources.configuration.locale
    val devLocale = c.resources.configuration.locales.get(0)
    return if (devLocale.language == "en" && devLocale.country == "DE")
        DateFormat.getDateInstance(DateFormat.SHORT, Locale.GERMANY)
    else DateFormat.getDateInstance(DateFormat.SHORT, Locale.getDefault())
}

fun millisecondsToLocalDate(m: Long): LocalDateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(m), ZoneOffset.UTC)
fun localDateToMilliseconds(localDate: LocalDate) = localDate.atStartOfDay(ZoneOffset.UTC).toInstant().toEpochMilli()

// Handle deprecation error
fun getVersionCode(context: Context): String {
    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) getLongVersionCode(context).toString()
    else getIntVersionCode(context).toString()
}

// Method to retrieve version code using PackageManager
@Suppress("DEPRECATION")
fun getIntVersionCode(context: Context): Int {
    try {
        val packageManager = context.packageManager
        val packageName = context.packageName
        val packageInfo = packageManager.getPackageInfo(packageName, 0)
        return packageInfo.versionCode
    } catch (e: PackageManager.NameNotFoundException) {
        e.printStackTrace()
        return -1 // Error occurred, return -1 or handle it accordingly
    }
}

@RequiresApi(Build.VERSION_CODES.P)
fun getLongVersionCode(context: Context): Long {
    try {
        val packageManager = context.packageManager
        val packageName = context.packageName
        val packageInfo = packageManager.getPackageInfo(packageName, 0)
        return packageInfo.longVersionCode
    } catch (e: PackageManager.NameNotFoundException) {
        e.printStackTrace()
        return -1L // Error occurred, return -1 or handle it accordingly
    }
}

// Method to retrieve version name using PackageManager
fun getVersionName(context: Context): String {
    try {
        val packageManager = context.packageManager
        val packageName = context.packageName
        val packageInfo = packageManager.getPackageInfo(packageName, 0)
        return packageInfo.versionName.toString()
    } catch (e: PackageManager.NameNotFoundException) {
        e.printStackTrace()
        return "" // Error occurred, return empty string or handle it accordingly
    }
}

fun isValidFilename(filename: String): Boolean {
    // Basic checks
    if (filename.isEmpty() || filename.length > 255) return false

    // Forbidden characters for Windows and cross-platform safety
    val forbidden = Regex("""[\\/:*?"<>|]""")
    if (forbidden.containsMatchIn(filename)) return false

    // Null character check (very rare in practice)
    if (filename.contains('\u0000')) return false

    // Optionally, disallow leading/trailing whitespace or dots
    if (filename.trim() != filename) return false
    if (filename == "." || filename == "..") return false

    return true
}

fun getCertificateSubject(context: Context): String {
    @Suppress("DEPRECATION") val sig =
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) context.packageManager.getPackageInfo(context.packageName, PackageManager.GET_SIGNING_CERTIFICATES).signingInfo?.apkContentsSigners[0]
        else context.packageManager.getPackageInfo(context.packageName, PackageManager.GET_SIGNATURES).signatures?.get(0)

    val certStream: InputStream = ByteArrayInputStream(sig?.toByteArray())
    return try {
        val certFactory = CertificateFactory.getInstance("X509")
        val x509Cert: X509Certificate = certFactory.generateCertificate(certStream) as X509Certificate
        x509Cert.subjectDN.toString()
    } catch (_: CertificateException) {
        "Unknown"
    }
}


// Checks if entered value is within the limits of the field
fun onValueChangeCheckIntValue(field: String, max: Int, updateValue: (String) -> Unit): Boolean {
    if (field.isEmpty())
        updateValue("") // To be able to clear the field
    else {
        val number = field.toIntOrNull()
        if (number != null && number in 1..max) updateValue(field)
    }

    // Jump if field is 3 digits or starts with >=3 and is 2 digits
    val maxLength = max.toString().length
    val firstDigit = max.toString()[0].digitToInt()+1 // If max= 199 -> this is 2
    val crit1 = field.length == maxLength
    val crit2 = (field.length == (maxLength-1) && field[0].digitToInt() >= firstDigit)
    return (crit1 || crit2)
}

fun getMillisMinusMonths(months: Int): Long {
    val now = ZonedDateTime.now(ZoneId.systemDefault())
    // Subtract months, then clamp day-of-month to the max valid for the target month
    val targetMonth = now.minusMonths(months.toLong())
    val adjustedDay = now.dayOfMonth.coerceAtMost(targetMonth.toLocalDate().lengthOfMonth())
    val targetDate = targetMonth.withDayOfMonth(adjustedDay)
    return targetDate.toInstant().toEpochMilli()
}

fun isImperial(): Boolean {
    // At this time, only three countries - Burma, Liberia, and the US - have not adopted the International System of Units (SI, or metric system) as their official system of weights and measures
    // Retrieve system language
    val country = Locale.getDefault().country
    return when (country) { // US, Liberia, Myanmar use imperial
        "LR",
        "MM",
        "US" -> true
        else -> false
    }
}

fun setDefaults(context: Context) {
    val preferences = PreferenceManager.getDefaultSharedPreferences(context)

    // At this time, only three countries - Burma, Liberia, and the US - have not adopted the International System of Units (SI, or metric system) as their official system of weights and measures
    // Retrieve system language
    val country = Locale.getDefault().country
    when (country) { // US, Liberia, Myanmar use imperial
        "LR",
        "MM",
        "US" -> {
            preferences.edit {
                putString(SettingsActivity.KEY_PREF_WEIGHT_PAPER_SIZE, "Letter")
                putString(SettingsActivity.KEY_PREF_BLOODPRESSURE_PAPER_SIZE, "Letter")
                putString(SettingsActivity.KEY_PREF_DIARY_PAPER_SIZE, "Letter")
                putString(SettingsActivity.KEY_PREF_TEMPERATURE_PAPER_SIZE, "Letter")
                putString(SettingsActivity.KEY_PREF_GLUCOSE_PAPER_SIZE, "Letter")
                putString(SettingsActivity.KEY_PREF_OXIMETRY_PAPER_SIZE, "Letter")
                putString(SettingsActivity.KEY_PREF_FLUID_PAPER_SIZE, "Letter")
                putString(SettingsActivity.KEY_PREF_WEIGHT_UNIT, "lbs")
                putString(SettingsActivity.KEY_PREF_WEIGHT_THRESHOLDS, context.getString(R.string.WEIGHT_THRESHOLD_DEFAULT_LBS))
                putString(SettingsActivity.KEY_PREF_FLUID_UNIT, "oz")
                putString(SettingsActivity.KEY_PREF_FLUID_THRESHOLD, "125")
            }
        }
        else -> {
            preferences.edit {
                putString(SettingsActivity.KEY_PREF_WEIGHT_PAPER_SIZE, context.getString(R.string.WEIGHT_PAPER_SIZE_DEFAULT))
                putString(SettingsActivity.KEY_PREF_BLOODPRESSURE_PAPER_SIZE, context.getString(R.string.BLOODPRESSURE_PAPER_SIZE_DEFAULT))
                putString(SettingsActivity.KEY_PREF_DIARY_PAPER_SIZE, context.getString(R.string.DIARY_PAPER_SIZE_DEFAULT))
                putString(SettingsActivity.KEY_PREF_TEMPERATURE_PAPER_SIZE, context.getString(R.string.TEMPERATURE_PAPER_SIZE_DEFAULT))
                putString(SettingsActivity.KEY_PREF_GLUCOSE_PAPER_SIZE, context.getString(R.string.GLUCOSE_PAPER_SIZE_DEFAULT))
                putString(SettingsActivity.KEY_PREF_OXIMETRY_PAPER_SIZE, context.getString(R.string.OXIMETRY_PAPER_SIZE_DEFAULT))
                putString(SettingsActivity.KEY_PREF_FLUID_PAPER_SIZE, context.getString(R.string.FLUID_PAPER_SIZE_DEFAULT))
                putString(SettingsActivity.KEY_PREF_WEIGHT_UNIT, context.getString(R.string.WEIGHT_UNIT_DEFAULT))
                putString(SettingsActivity.KEY_PREF_WEIGHT_THRESHOLDS, context.getString(R.string.WEIGHT_THRESHOLD_DEFAULT_KG))
                putString(SettingsActivity.KEY_PREF_FLUID_UNIT, context.getString(R.string.FLUID_UNIT_DEFAULT))
                putString(SettingsActivity.KEY_PREF_FLUID_THRESHOLD, context.getString(R.string.FLUID_THRESHOLD_DEFAULT))
            }
        }
    }
}
