/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.receiver

import android.content.Intent
import android.content.Intent.EXTRA_STREAM
import android.content.SharedPreferences
import android.net.Uri
import android.os.Bundle
import android.text.Spannable
import android.text.SpannableString
import android.text.style.ClickableSpan
import android.text.style.ForegroundColorSpan
import android.view.View
import android.widget.Toast
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.edit
import androidx.core.graphics.toColorInt
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.ACTIVE_PROFILE_KEY
import com.zell_mbc.medilog.ACTIVE_TAB_KEY
import com.zell_mbc.medilog.ActiveProfile
import com.zell_mbc.medilog.MainActivity.Companion.DatePattern
import com.zell_mbc.medilog.Tabs
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.UNSET
import com.zell_mbc.medilog.data.Data
import com.zell_mbc.medilog.diary.DiaryViewModel
import com.zell_mbc.medilog.documents.DocumentsViewModel
import com.zell_mbc.medilog.profiles.ProfilesViewModel
import com.zell_mbc.medilog.scaffold.Screens
import com.zell_mbc.medilog.settings.SettingsViewModel
import com.zell_mbc.medilog.support.getAttachmentFolder
import com.zell_mbc.medilog.tags.TagsViewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import java.io.File
import java.io.InputStream
import java.net.URL
import java.text.SimpleDateFormat
import java.util.Date
import java.util.regex.Pattern
import javax.net.ssl.HttpsURLConnection
import kotlin.getValue

class ReceiverActivity: AppCompatActivity() {
    val documentsViewModel: DocumentsViewModel by viewModels()
    val profilesViewModel: ProfilesViewModel by viewModels()
    val diaryViewModel: DiaryViewModel by viewModels()
    val tagsViewModel: TagsViewModel by viewModels()

    private lateinit var preferences: SharedPreferences

    // https://anant-raman.medium.com/extract-hyperlinks-and-emails-from-a-text-and-perform-actions-in-android-kotlin-5483fe74da5e
    private val urlPattern: Pattern = Pattern.compile(
        "(?:^|[\\W])((ht|f)tp(s?):\\/\\/|www\\.)"
                + "(([\\w\\-]+\\.){1,}?([\\w\\-.~]+\\/?)*"
                + "[\\p{Alnum}.,%_=?&#\\-+()\\[\\]\\*$~@!:/{};']*)",
        Pattern.CASE_INSENSITIVE or Pattern.MULTILINE or Pattern.DOTALL
    )

    private val emailPattern: Pattern = Pattern.compile(
        "[a-zA-Z0-9\\+\\.\\_\\%\\-\\+]{1,256}" +
                "\\@" +
                "[a-zA-Z0-9][a-zA-Z0-9\\-]{0,64}" +
                "(" +
                "\\." +
                "[a-zA-Z0-9][a-zA-Z0-9\\-]{0,25}" +
                ")+"
    )

    //Function to extract hyperlinks from a given stringprivate
    fun getHyperLinks(s: String): List<Pair<Int, Int>> {
        val urlList = mutableListOf<Pair<Int, Int>>()
        val urlMatcher = urlPattern.matcher(s)
        var matchStart: Int
        var matchEnd: Int
        while (urlMatcher.find()) {
            matchStart = urlMatcher.start(1)
            matchEnd = urlMatcher.end()
            urlList.add(Pair(matchStart, matchEnd))
            val url = s.substring(matchStart, matchEnd)
        }
        return urlList
    }

    //Function to extract emails from a given string
    private fun getEmailLists(s: String): List<Pair<Int, Int>> {
        val emailList = mutableListOf<Pair<Int, Int>>()
        val emailMatcher = emailPattern.matcher(s)
        while (emailMatcher.find()) {
            val email = emailMatcher.group()
            emailList.add(Pair(emailMatcher.start(), emailMatcher.start() + email.length))
        }
        return emailList
    }


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        // We can't expect ActiveProfile to be set
        preferences = PreferenceManager.getDefaultSharedPreferences(this)
        ActiveProfile.id = preferences.getInt(ACTIVE_PROFILE_KEY, UNSET)

        if (intent?.action == Intent.ACTION_SEND)
            when(intent.type) {
               "text/plain" -> handleSendText(intent) // Handle text being sent
               "application/pdf" -> handlePdf(intent) // Handle text being sent
               "image/png" -> handleFile(intent, "png") // Handle text being sent
               "image/jpeg" -> handleFile(intent, "jpg") // Handle text being sent
        }
    }

    //Function to customise texts which are identifed as a hyperlink or an email
    private fun customiseText(spanStr: SpannableString, start: Int, end: Int): SpannableString {
        val clickSpan = object : ClickableSpan() {
            override fun onClick(widget: View) {
                // Write the actions you want to be performed on click of the particular hyperlink or email
            }
        }
        spanStr.setSpan(clickSpan, start, end, Spannable.SPAN_INCLUSIVE_EXCLUSIVE)//Change the colour of the hyperlink or the email
        spanStr.setSpan(ForegroundColorSpan("#FF39ACEE".toColorInt()), start, end, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE)
        return spanStr
    }

    private fun extractUrl(source: String): String {
        val urlMatcher = urlPattern.matcher(source)
        var matchStart: Int
        var matchEnd: Int
        return if (urlMatcher.find()) {
            matchStart = urlMatcher.start(1)
            matchEnd = urlMatcher.end()
            source.substring(matchStart, matchEnd)
        }
        else ""
    }

    fun extractSigmaData(urlString: String): String {
        var retValue = ""

        runBlocking {
            val j = launch(Dispatchers.IO) {
                val url = URL(urlString)
                val urlConnection = url.openConnection() as HttpsURLConnection
                try {
                    val website = urlConnection.inputStream.bufferedReader().readText()
                    val searchText = "<meta name='description' content='Name: "
                    val index = website.indexOf(searchText)
                    if (index >= 0) {
                        val newStart = website.substring(index + searchText.length)
                        val index = newStart.indexOf("\n")
                        retValue = newStart.substring(0, index)
                    }
                } finally {
                    urlConnection.disconnect()
                }
            }
            j.join()
        }
        return retValue
    }

    private fun handleSendText(intent: Intent) {
        intent.getStringExtra(Intent.EXTRA_TEXT)?.let {
            /*val listOfUrls = getHyperLinks(it)
            val listOfEmails = getEmailLists(it)
            val spanPoint = SpannableString(it)
            var point: SpannableString? = null
            for (url in listOfUrls) {
                point = customiseText(spanPoint, url.first, url.second)
            }
            for (email in listOfEmails) {
                point = customiseText(spanPoint, email.first,   email.second)
            }*/

            val sportLabel = "Sport"
            var tagId = ""
            var text = it.toString()

            val url = extractUrl(text)
            if (url.isNotEmpty()) {
                if (url.contains("sigma-sharing.com")) {
                    val extract = extractSigmaData(url)
                    if (extract.isNotEmpty()) {
                        text = "$extract, $url"
                        tagId = tagsViewModel.searchTag(sportLabel)
                    }
                }
            }
            val item = Data(0, Date().time, text, Tabs.DIARY, value1 = "", value2 = "0", value3 = "", value4 = "", attachment = "", tags = tagId, category_id = -1, profile_id = ActiveProfile.id)
            diaryViewModel.upsertBlocking(item) // Blocking to make sure data is written before Activity is killed

            preferences.edit { putString(ACTIVE_TAB_KEY, Screens.Diary.route) }

            val message = getString(R.string.receivedDiaryEntry) + ": $text"
            Toast.makeText(this, message, Toast.LENGTH_LONG).show()

            finish()

            /* Todo
             Receiver Settings screen:
             - Open MediLog yes/no
             - Assign tag to sigma-sharing
             */
        }
    }

    private fun handlePdf(intent: Intent) {
        val pdfUri = intent.extras?.get(EXTRA_STREAM) as Uri

        val filenameString = pdfUri.toString()
        if (filenameString.lowercase().contains(".pdf")) {
            val fileName = filenameString.substring(filenameString.lastIndexOf("/") +1)
            val newFile = File(getAttachmentFolder(this), fileName)

            val resolver = contentResolver
            val inputStream: InputStream? = resolver.openInputStream(pdfUri)
            inputStream.use { input -> newFile.outputStream().use { output -> input!!.copyTo(output) } }


            val item = Data(0, timestamp = Date().time, comment = fileName, type = Tabs.DOCUMENTS, value1 = "", value2 = "", value3 = "", value4 = "", attachment = fileName, tags ="", category_id = -1, profile_id = ActiveProfile.id)
            documentsViewModel.upsertBlocking(item)

            preferences.edit { putString(ACTIVE_TAB_KEY, Screens.Documents.route) }

            val message = getString(R.string.receivedDocument) + ": $fileName"
            Toast.makeText(this, message, Toast.LENGTH_LONG).show()

            finish()
        }
        /*val intent = Intent(this, MainActivity::class.java)
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        startActivity(intent)*/
    }

    private fun handleFile(intent: Intent, extension: String) {
        val imageUri = intent.extras?.get(EXTRA_STREAM) as Uri

        val filenameString = imageUri.toString()
        val fileName = getString(string.appName) + "-" + SimpleDateFormat(DatePattern.DATE_TIME).format(Date().time) + ".$extension"
        val newFile = File(getAttachmentFolder(this), fileName)

        val resolver = contentResolver
        val inputStream: InputStream? = resolver.openInputStream(imageUri)
        inputStream.use { input -> newFile.outputStream().use { output -> input!!.copyTo(output) } }

        val item = Data(0, timestamp = Date().time, comment = fileName, type = Tabs.DIARY, value1 = "", value2 = "", value3 = "", value4 = "", attachment = fileName, tags ="", category_id = -1, profile_id = ActiveProfile.id)
        diaryViewModel.upsertBlocking(item)

        preferences.edit { putString(ACTIVE_TAB_KEY, Screens.Documents.route) }
        val message = getString(R.string.receivedDocument) + ": $fileName"
        Toast.makeText(this, message, Toast.LENGTH_LONG).show()

        finish()
    }


}