/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.profiles

import android.content.Context
import android.content.SharedPreferences
import android.util.Log
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.PROFILE_PREFERENCES_STUB
import androidx.core.content.edit
import com.zell_mbc.medilog.ACTIVE_PROFILE_KEY
import com.zell_mbc.medilog.settings.SettingsViewModel

object PreferenceSync {
    private const val PROFILE_PREFIX = "profile_"

    fun switchProfile(context: Context, oldProfileId: Int, newProfileId: Int) {
        val masterPrefs = PreferenceManager.getDefaultSharedPreferences(context)

        val oldProfilePrefs = context.getSharedPreferences(
            PROFILE_PREFERENCES_STUB + oldProfileId, Context.MODE_PRIVATE
        )
        val newProfilePrefs = context.getSharedPreferences(
            PROFILE_PREFERENCES_STUB + newProfileId, Context.MODE_PRIVATE
        )

        // 1️⃣ Save current profile-level values
        oldProfilePrefs.edit {
            for ((key, value) in masterPrefs.all) {
                if (key.startsWith(PROFILE_PREFIX)) {
                    putAny(this, key, value)
                }
            }
        }

        // 2️⃣ Remove old profile values from master
        masterPrefs.edit {
            for (key in masterPrefs.all.keys) {
                if (key.startsWith(PROFILE_PREFIX)) remove(key)
            }

            // 3️⃣ Blend app-level + new profile-level values into master
            for ((key, value) in newProfilePrefs.all) {
                if (key.startsWith(PROFILE_PREFIX)) {
                    putAny(this, key, value)
                }
            }
        }

        // 4️⃣ Update active profile ID
        masterPrefs.edit { putInt(ACTIVE_PROFILE_KEY, newProfileId) } // No need to also set ActiveProfile here because the parent calling this function will reload MainActivity anyway

        Log.i("PreferenceSync", "Profile switched to $newProfileId")
    }

    private fun putAny(editor: SharedPreferences.Editor, key: String, value: Any?) {
        when (value) {
            is Boolean -> editor.putBoolean(key, value)
            is Int -> editor.putInt(key, value)
            is Float -> editor.putFloat(key, value)
            is Long -> editor.putLong(key, value)
            is String -> editor.putString(key, value)
            else -> editor.remove(key)
        }
    }

    // Migrate values of active profile
    fun migrateLegacyProfilePrefs(context: Context) {
        val legacyProfileKeys = setOf(
            "etHeight",
            "etWeightThreshold",
            "cbLogBodyFat",
            "etFatMinMax",
            "etTare",
            "etTemperatureThresholds",
            "etOximetryThresholds",
            "etGlucoseThresholds",
            "evFluidThreshold",
            "activeTabs",
            "activeTab"
        )

        val masterPrefs = PreferenceManager.getDefaultSharedPreferences(context)
        val alreadyMigrated = masterPrefs.getBoolean("preference_migration_done", false)

        if (alreadyMigrated) return // nothing to do

        masterPrefs.edit {
            // For every legacy key, copy it into a new "profile_"-prefixed key
            for (key in legacyProfileKeys) {
                val value = masterPrefs.all[key]
                if (value != null) {
                    val newKey = "profile_$key" + if (key == "etWeightThreshold") "s" else "" // Changing name from singular to plural :-/
                    when (value) {
                        is Boolean -> putBoolean(newKey, value)
                        is Int -> putInt(newKey, value)
                        is Float -> putFloat(newKey, value)
                        is Long -> putLong(newKey, value)
                        is String -> putString(newKey, value)
                    }
                    remove(key)
                }
            }
            putBoolean("preference_migration_done", true)
        }
    }
}
