/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.preferences
import android.os.Bundle
import android.text.InputFilter
import android.text.InputType
import android.text.Spanned
import android.widget.Toast
import androidx.preference.EditTextPreference
import androidx.preference.ListPreference
import androidx.preference.Preference
import androidx.preference.PreferenceFragmentCompat
import androidx.preference.PreferenceManager
import androidx.preference.SwitchPreference
import com.zell_mbc.medilog.MainActivity
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_WEIGHT_CUSTOM_TEMPLATE
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_WEIGHT_TARE
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_WEIGHT_THRESHOLDS
import com.zell_mbc.medilog.support.TemplateInputFilter
import com.zell_mbc.medilog.support.isImperial

class WeightSettingsFragment : PreferenceFragmentCompat() {
    override fun onCreatePreferences(savedInstanceState: Bundle?, rootKey: String?) {
        setPreferencesFromResource(R.xml.weight_preferences, rootKey)

        // Fix an Android bug: https://developer.android.com/develop/ui/views/components/settings/customize-your-settings#customize_an_edittextpreference_dialog
        val weightUnit = preferenceManager.findPreference<ListPreference>(SettingsActivity.KEY_PREF_WEIGHT_UNIT)
        weightUnit?.summary = requireContext().getString(R.string.weightUnitSummary) + ", " + weightUnit.value
        weightUnit?.onPreferenceChangeListener = Preference.OnPreferenceChangeListener { _, newValue ->
            val tmpString = weightUnit.summary.toString()
            weightUnit.summary = tmpString.replace(weightUnit.value.toString(), newValue.toString())
            true
        }

        val tare = preferenceManager.findPreference<EditTextPreference>(KEY_PREF_WEIGHT_TARE)
        tare?.setOnBindEditTextListener { editText -> editText.inputType = InputType.TYPE_CLASS_NUMBER + InputType.TYPE_NUMBER_FLAG_DECIMAL}
        tare?.summary = requireContext().getString(R.string.tareSummary) + if (tare.text?.isNotEmpty() == true) ", " + tare.text + " " + weightUnit?.value else ""
        tare?.onPreferenceChangeListener = Preference.OnPreferenceChangeListener { _, newValue ->
            try {
                if (!newValue.toString().isEmpty()) {
                    val tmpValue = newValue.toString().toFloat()
                    tare.summary = getString(R.string.tareSummary) + ", $tmpValue " + weightUnit?.value
                }
                else tare.summary = requireContext().getString(R.string.tareSummary)
                true
            }
            catch (_: NumberFormatException) {
                Toast.makeText(requireContext(), requireContext().getString(R.string.invalidInput), Toast.LENGTH_LONG).show()
                false
            }
        }

        //-------------------------------------------------
        class ThresholdsInputFilter : InputFilter {
            override fun filter(source: CharSequence, start: Int, end: Int, dest: Spanned, dstart: Int, dend: Int): CharSequence? {
                // Build the new text as if the input is accepted
                val newText = (dest.substring(0, dstart) + source + dest.substring(dend)).replace(',', '.')

                // Split at "/"
                val parts = newText.split("-")
                val first = parts.getOrNull(0) ?: ""
                val second = parts.getOrNull(1) ?: ""

                // check valiD float
                first.toFloatOrNull() ?: return ""

                // Check first part has got only one decimal
                val i1 = first.indexOf(".")
                if (i1 > 0 && first.length > i1 + 2) return ""

                // Check second part
                if (second.isNotEmpty()) {
                    second.toFloatOrNull() ?: return ""

                    // Check second part has got only one decimal
                    val i2 = second.indexOf(".")
                    if (i2> 0 && second.length > i2 + 2) return ""

                }

                return null // accept
            }
        }

        val thresholdObject = preferenceManager.findPreference<EditTextPreference>(KEY_PREF_WEIGHT_THRESHOLDS)
        thresholdObject?.setOnBindEditTextListener { editText ->
            val filterArray = arrayOfNulls<InputFilter>(1)
            //filterArray[0] = InputFilter.LengthFilter(7)
            filterArray[0] = ThresholdsInputFilter()
            editText.filters = filterArray
        }

        // Initialize default if empty
        if (thresholdObject?.text.isNullOrEmpty()) {
            thresholdObject?.text = if (isImperial()) getString(R.string.WEIGHT_THRESHOLD_DEFAULT_LBS)
            else getString(R.string.WEIGHT_THRESHOLD_DEFAULT_KG)
        }

        // Update summary function
        thresholdObject?.summary = requireContext().getString(R.string.thresholdSummaryRange) + ", " + thresholdObject.text + " " + weightUnit?.value

        // Validate when preference is saved
        thresholdObject?.onPreferenceChangeListener = Preference.OnPreferenceChangeListener { _, newValue ->
            val valueStr = newValue.toString().trim().replace(',', '.') // normalize decimal
            val parts = valueStr.split("-")
            if (parts.size != 2) {
                Toast.makeText(requireContext(), getString(R.string.thresholdsInvalidFormatHint, "00.0-000.0"), Toast.LENGTH_LONG).show()
                return@OnPreferenceChangeListener false
            }

            try {
                val lower = parts[0].toFloat()
                val upper = parts[1].toFloat()

                if (lower >= upper) {
                    Toast.makeText(requireContext(), getString(R.string.thresholdsInvalidRange), Toast.LENGTH_LONG).show()
                    return@OnPreferenceChangeListener false
                }

                // Update summary function
                thresholdObject.summary = requireContext().getString(R.string.thresholdSummaryRange) + ", " + newValue + " " + weightUnit?.value
            } catch (_: NumberFormatException) {
                Toast.makeText(requireContext(), getString(R.string.thresholdsInvalidFormatHint, "0.0-000.0"), Toast.LENGTH_LONG).show()
                return@OnPreferenceChangeListener false
            }

            true
        }

//-------------------------------------------------

       val paperSize = preferenceManager.findPreference<ListPreference>(SettingsActivity.KEY_PREF_WEIGHT_PAPER_SIZE)
        paperSize?.summary = requireContext().getString(R.string.paperSizeSummary) + ", " + paperSize.value
        paperSize?.onPreferenceChangeListener = Preference.OnPreferenceChangeListener { _, newValue ->
            val tmpString = paperSize.summary.toString()
            paperSize.summary = tmpString.replace(paperSize.value.toString(), newValue.toString())
            true
        }

//-------------------------------------------------
        val bodyHeight = preferenceManager.findPreference<EditTextPreference>(SettingsActivity.KEY_PREF_BODY_HEIGHT)
        bodyHeight?.setOnBindEditTextListener { editText -> editText.inputType = InputType.TYPE_CLASS_NUMBER }
        bodyHeight?.summary = requireContext().getString(R.string.heightSummary) + if (bodyHeight.text?.isNotEmpty() == true) ", " + bodyHeight.text else ""
        bodyHeight?.onPreferenceChangeListener = Preference.OnPreferenceChangeListener { _, newValue ->
            val tmpString = requireContext().getString(R.string.invalidBodyHeight)
            try {
                if (!newValue.toString().isEmpty()) {
                    newValue.toString().toInt()
                    bodyHeight.summary = getString(R.string.heightSummary) + ", $newValue"
                }
                else bodyHeight.summary = getString(R.string.heightSummary)
                true
            }
            catch (_: NumberFormatException) {
                Toast.makeText(requireContext(), tmpString, Toast.LENGTH_LONG).show()
                false
            }
        }

        //-------------------------------------------------
        val preferences = PreferenceManager.getDefaultSharedPreferences(requireContext())
        preferenceManager.findPreference<Preference>(SettingsActivity.KEY_PREF_WEIGHT_MOVING_AVERAGE_SIZE)?.isEnabled = preferences.getBoolean(SettingsActivity.KEY_PREF_WEIGHT_SHOW_MOVING_AVERAGE, false)

        val movingAverage = preferenceManager.findPreference<SwitchPreference>(SettingsActivity.KEY_PREF_WEIGHT_SHOW_MOVING_AVERAGE)
        movingAverage?.onPreferenceChangeListener = Preference.OnPreferenceChangeListener { _, newValue ->
            preferenceManager.findPreference<Preference>(SettingsActivity.KEY_PREF_WEIGHT_MOVING_AVERAGE_SIZE)?.isEnabled = newValue.toString().toBoolean()
            true
        }

        val inputFormatObject = preferenceManager.findPreference<EditTextPreference>(KEY_PREF_WEIGHT_CUSTOM_TEMPLATE)
        inputFormatObject?.setOnBindEditTextListener { editText ->
            val filterArray = arrayOfNulls<InputFilter>(2)
            filterArray[0] = InputFilter.LengthFilter(7)
            filterArray[1] = TemplateInputFilter(context)
            editText.filters = filterArray
        }

    }

    override fun onPause() {
        super.onPause()
        MainActivity.resetReAuthenticationTimer(requireContext())
    }
}