/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.glucose

import android.content.Context
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.text.KeyboardActions
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Abc
import androidx.compose.material3.BasicAlertDialog
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.RadioButton
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.VerticalDivider
import androidx.compose.runtime.Composable
import androidx.compose.runtime.SideEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusDirection
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.PlatformTextStyle
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.input.KeyboardCapitalization
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.zell_mbc.medilog.MainActivity
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.base.BaseTab
import com.zell_mbc.medilog.data.Data
import com.zell_mbc.medilog.support.SnackbarDelegate
import java.util.Date
import androidx.core.content.edit
import com.zell_mbc.medilog.ActiveProfile
import com.zell_mbc.medilog.Filter
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.Thresholds
import com.zell_mbc.medilog.glucoseUnitUndefined
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_GLUCOSE_UNIT
import com.zell_mbc.medilog.preferences.SettingsActivity.Companion.KEY_PREF_QUICKENTRY
import com.zell_mbc.medilog.scaffold.Screens
import com.zell_mbc.medilog.support.checkMinMax
import com.zell_mbc.medilog.support.matchesTemplate
import com.zell_mbc.medilog.support.shouldMoveFocus
import com.zell_mbc.medilog.tags.TagsViewModel

enum class GlucoseUnit(val labelResId: Int) {
    MG_DL(string.GLUCOSE_UNIT_MG_DL),
    MMOL_L(string.GLUCOSE_UNIT_MMOL_L)
}

class GlucoseTab(override val label: String?, override val viewModel: GlucoseViewModel, override val tagsViewModel: TagsViewModel, context: Context, snackbarDelegate: SnackbarDelegate): BaseTab(viewModel, tagsViewModel, context, snackbarDelegate) {
    override var editActivityClass: Class<*> = GlucoseEditActivity::class.java
    override var infoActivityClass: Class<*> = GlucoseInfoActivity::class.java
    override var chartActivityClass: Class<*> = GlucoseChartActivity::class.java

    override val route = Screens.Glucose.route
    override val fabEnabled = true
    override val iconActive = R.drawable.ic_glucose_filled
    override val iconInactive = R.drawable.ic_glucose_outlined

    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    override fun ShowContent(padding: PaddingValues) {
        super.ShowContent(padding)

        // Global variables

        val showChangeUnitDialog = remember { mutableStateOf(false) }
        var showComment by remember { mutableStateOf(viewModel.comment.value.isNotEmpty()) }
        value1Width = measureTextWidth(viewModel.value1.template, MainActivity.fontSize)
        value2Width = measureTextWidth(viewModel.value2.template, MainActivity.fontSize)
        value3Width = measureTextWidth(viewModel.value3.template, MainActivity.fontSize)

        LazyColumn(state = listState, horizontalAlignment = Alignment.Start, modifier = Modifier.fillMaxWidth().padding(start = 8.dp, end = 8.dp)) {
            item {
                if (viewModel.quickEntry) {
                    // Glucose
                    val focusManager = LocalFocusManager.current
                    val (textField1, textField2, textField3) = remember { FocusRequester.createRefs() }
                    Row(Modifier.fillMaxWidth()) {
                        TextField(
                            value = viewModel.value1.value,
                            colors = textFieldColors,
                            onValueChange = {
                                if (!viewModel.logKetone) showComment = true // only activate comment field if logKetone is not visible
                                // 1. Check template
                                if (matchesTemplate(it, viewModel.value1.template)) {
                                    // 2. Max/min reached?
                                    val max = if (viewModel.mg_dl) Thresholds.MAX_GLUCOSE_MG else Thresholds.MAX_GLUCOSE_MMOL.toInt()
                                    if (checkMinMax(field = it, min = 0, max = max)) {
                                        viewModel.value1.value = it // Single source of truth!
                                        // 3. Jump if needed
                                        if (shouldMoveFocus(it, viewModel.value1.template, max)) focusManager.moveFocus(FocusDirection.Next)
                                    }
                                }

                            },
                            singleLine = true,
                            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                            textStyle = TextStyle(),
                            label = { Text(text = "${viewModel.value1.label}*", maxLines = 1, overflow = TextOverflow.Ellipsis) },
                            placeholder = { Text("${viewModel.value1.template} ${viewModel.value1.unit}") },
                            modifier = Modifier.weight(1f).focusRequester(textField1).padding(end = 10.dp),
                            keyboardActions = KeyboardActions(onDone = { addItem() })
                        )
                        if (viewModel.logKetone)
                            TextField(
                                value = viewModel.value2.value,
                                colors = textFieldColors,
                                onValueChange = {
                                    showComment = true // only activate comment field if logKetone is not visible
                                    if (shouldMoveFocus(it, viewModel.value2.template)) focusManager.moveFocus(FocusDirection.Next)
                                    if (matchesTemplate(it, viewModel.value2.template)) viewModel.value2.value = it // Single source of truth!
                                },
                                singleLine = true,
                                keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                                textStyle = TextStyle(),
                                label = { Text(text = "${viewModel.value2.label}*", maxLines = 1, overflow = TextOverflow.Ellipsis) },
                                placeholder = { Text("${viewModel.value2.template} ${viewModel.value2.unit}") },
                                modifier = Modifier.weight(1f).focusRequester(textField2).padding(end = cellPadding.dp),
                                keyboardActions = KeyboardActions( onDone = { addItem() })
                            )
                    } // Row

                    if (showComment)
                        TextField(
                            value = viewModel.comment.value,
                            modifier = Modifier.fillMaxWidth().focusRequester(textField3),
                            colors = textFieldColors,
                            onValueChange = { viewModel.comment.value = it },
                            singleLine = true,
                            textStyle = TextStyle(),
                            label = { Text(text = viewModel.comment.label) },
                            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Text, capitalization = KeyboardCapitalization.Sentences),
                            trailingIcon = { IconButton(onClick = { openTextTemplatesDialog.value = true }) { Icon(Icons.Outlined.Abc, contentDescription = null) } },
                            keyboardActions = KeyboardActions(onDone = {
                                keyboardController?.hide()
                                addItem()
                            })
                        )
                    SideEffect {
                        // Set cursor to first field after addItem completed
                        if (activateFirstField) {
                            textField1.requestFocus()
                            activateFirstField = false }
                    }

                    // Dialog section
                    if (openTextTemplatesDialog.value) {
                        textTemplateDialog.ShowDialog(viewModel.dataType, setShowDialog = { openTextTemplatesDialog.value = it }) {
                            viewModel.comment.value += it
                            textField3.requestFocus()
                        }
                    }

                    // Is the unit already set? If no confirm unit
                    if (viewModel.value1.unit.equals(glucoseUnitUndefined)) showChangeUnitDialog.value = true
                    if (showChangeUnitDialog.value) {
                        BasicAlertDialog(onDismissRequest = { showChangeUnitDialog.value = false }) {
                            var selectedUnit by remember { mutableStateOf(GlucoseUnit.MG_DL) }

                            Surface(shape = MaterialTheme.shapes.large, modifier = Modifier.fillMaxWidth(.95f)) {
                                Column(modifier = Modifier.padding(16.dp)) {
                                    val glucose = stringResource(string.glucose)
                                    Text(stringResource(string.unitDialogHeader), style = MaterialTheme.typography.headlineMedium)
                                    Text("")
                                    Text(stringResource(string.unitDialogSubtitle, glucose), style = MaterialTheme.typography.bodyMedium)
                                    Text("")
                                    Row {
                                        GlucoseUnit.entries.forEach { unit ->
                                            RadioButton(
                                                selected = selectedUnit == unit,
                                                onClick = { selectedUnit = unit }
                                            )
                                            Text(stringResource(unit.labelResId))
                                        }
                                    }
                                    Text("")
                                    Text(stringResource(string.unitDialogLine1), style = MaterialTheme.typography.bodySmall)
                                    Text(stringResource(string.unitDialogLine3) + " " + stringResource(GlucoseUnit.MG_DL.labelResId), style = MaterialTheme.typography.bodySmall)
                                    Text(stringResource(string.unitDialogLine2) + " " + stringResource(GlucoseUnit.MMOL_L.labelResId), style = MaterialTheme.typography.bodySmall)

                                    Row(modifier = Modifier.align(Alignment.End)) {
                                        OutlinedButton(onClick = {
                                            showChangeUnitDialog.value = false
                                            // Set Unit
                                            val idToString = context.getString(selectedUnit.labelResId)
                                            preferences.edit { putString(KEY_PREF_GLUCOSE_UNIT, idToString) }
                                            viewModel.setupEntryFields()
                                        }) { Text(stringResource(string.ok)) }
                                    }
                                }
                            }
                        }
                    }
                } // End quick entry
            } // item
            items(dataList.value, key = { it._id }) { item ->
                // Lines starting from the top
                HorizontalDivider(color = MaterialTheme.colorScheme.secondaryContainer, thickness = 1.dp)
                Row(modifier = Modifier.fillMaxWidth().height(IntrinsicSize.Min).clickable { selection.value = item }, verticalAlignment = Alignment.CenterVertically) {
                    ShowRow(item, value1Width)
                }
                if (selection.value != null) {
                    ItemClicked(selection.value!!._id)
                    selection.value = null
                }
            }
        }
    }

    @Composable
    fun ShowRow(item: Data, value1Width: Dp) {
        // Timestamp
        ResizableDateColumn(formatDateString(item.timestamp))

        var fontStyle = FontStyle.Normal
        var textColor = MaterialTheme.colorScheme.primary
        var newValue1String: String

        // A diary item blended in
        if (item.type != viewModel.dataType) {
            fontStyle = FontStyle.Italic
            val s = item.comment + if (item.attachment.isNotEmpty()) " $PAPERCLIP" else ""
            Text(s, Modifier.padding(start = cellPadding.dp), color = MaterialTheme.colorScheme.primary, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
            return
        }

        val newValue1Value = try { item.value1.toFloat() } catch (_: NumberFormatException) { 0f }

        // If unit cut off trailing .0
        newValue1String = if (viewModel.mg_dl) newValue1Value.toInt().toString() else newValue1Value.toString()
        if (MainActivity.modifyDecimalSeparator) newValue1String = newValue1String.replace('.', MainActivity.decimalSeparator)
        if (viewModel.highlightValues && ((newValue1Value < viewModel.lowerThreshold) || (newValue1Value > viewModel.upperThreshold))) textColor = MaterialTheme.colorScheme.error

        // Glucose
        Text(newValue1String, Modifier.padding(start = cellPadding.dp, end = cellPadding.dp).width(value1Width),
            textAlign = TextAlign.Center, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
        VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer, modifier = Modifier.width(1.dp).fillMaxHeight()) // Vertical separator

        // Ketone
        textColor = MaterialTheme.colorScheme.primary
        if (viewModel.logKetone) {
            var newValue2String = ""
            if (newValue1String.isNotEmpty()) {
                newValue2String = if (MainActivity.modifyDecimalSeparator) item.value2.replace('.', MainActivity.decimalSeparator) else item.value2
            }
            Text(newValue2String,
                Modifier.padding(start = cellPadding.dp, end = cellPadding.dp).width(value2Width),
                textAlign = TextAlign.Center, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
            VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer, modifier = Modifier.width(1.dp).fillMaxHeight()) // Vertical separator

            textColor = MaterialTheme.colorScheme.primary
            if (newValue2String.isNotEmpty()) {
                val gki = getGKI(item.value1, item.value2, viewModel.mg_dl)
                newValue2String = if (MainActivity.modifyDecimalSeparator) gki.replace('.', MainActivity.decimalSeparator) else gki
            }
            Text(newValue2String, Modifier.padding(start = cellPadding.dp, end = cellPadding.dp).width(value3Width),
                textAlign = TextAlign.Center, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle)
            VerticalDivider(color = MaterialTheme.colorScheme.secondaryContainer, modifier = Modifier.width(1.dp).fillMaxHeight()) // Vertical separator
        }

        // Comment
        Spacer(modifier = Modifier.width(cellPadding.dp))
        val s = item.comment + if (item.attachment.isNotEmpty()) " $PAPERCLIP" else ""
        Text(s, color = textColor, fontSize = MainActivity.fontSize.sp, fontStyle = fontStyle,
            style = TextStyle(lineHeight = TextUnit.Unspecified, platformStyle = PlatformTextStyle(includeFontPadding = false)) // style is for removing the padding between multiline text
        )
    }

    override fun addItem() {
        if (!preferences.getBoolean(KEY_PREF_QUICKENTRY, true)) {
            startEditing(-1) // Indicate that this is about a new item
            return
        }

        // Close keyboard after entry is done
        hideKeyboard()

        var validInput = matchesTemplate(viewModel.value1)
        if (!validInput) {
            val message =  context.getString(string.invalidFormat) + "\n" + context.getString(string.configuredFormat, viewModel.value1.template)
            snackbarDelegate.showSnackbar(message)
            return
        }
        validInput = matchesTemplate(viewModel.value2)
        if (!validInput) {
            val message =  context.getString(string.invalidFormat) + "\n" + context.getString(string.configuredFormat, viewModel.value1.template)
            snackbarDelegate.showSnackbar(message)
            return
        }

        val item = Data(_id = 0, timestamp = Date().time, comment = viewModel.comment.value, type = viewModel.dataType,
            value1 = viewModel.value1.value.replace(",","."),
            value2 = viewModel.value2.value.replace(",","."),
            value3 = "", value4 = "", attachment = "", tags = Filter.NO_TAGS, category_id = -1, profile_id = ActiveProfile.id) // Start with empty item
        viewModel.upsert(item)

        if ((viewModel.filterEnd > 0L) && (viewModel.filterEnd < item.timestamp))
            snackbarDelegate.showSnackbar(context.getString(string.filteredOut))

        cleanUpAfterAddItem()
    }
}