/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.fluid

import android.annotation.SuppressLint
import android.os.Bundle
import androidx.activity.compose.setContent
import androidx.activity.viewModels
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Abc
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.TextFieldColors
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusDirection
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.input.KeyboardCapitalization
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import com.zell_mbc.medilog.MainActivity.Companion.DAY_FORMAT
import com.zell_mbc.medilog.Tabs
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.base.EditActivity
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.support.matchesTemplate
import com.zell_mbc.medilog.support.shouldMoveFocus
import com.zell_mbc.medilog.tags.TagRows
import com.zell_mbc.medilog.tags.TagsViewModel
import java.text.SimpleDateFormat
import kotlin.getValue

class FluidEditActivity: EditActivity() {
    override val viewModel: FluidViewModel by viewModels()
    private val tagsViewModel: TagsViewModel by viewModels()

    override val dataType = Tabs.FLUID
    var enableTags = false

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        enableTags      = preferences.getBoolean(SettingsActivity.KEY_PREF_FLUID_ENABLE_TAGS, false)

        // All preparation done, start Compose
        setContent { StartCompose() }
    }

    @Composable
    override fun ShowContent() {
        Column(modifier = Modifier.padding(start = 8.dp, end = 8.dp)) {
            DateTimeBlock()

            val textFieldColors: TextFieldColors = TextFieldDefaults.colors(unfocusedContainerColor = Color.Transparent, focusedContainerColor = Color.Transparent)
            val focusManager = LocalFocusManager.current
            val (textField1, textField2) = remember { FocusRequester.createRefs() }

            if (enableTags)
                Row(Modifier.fillMaxWidth(), verticalAlignment = Alignment.CenterVertically) {
                    Column(modifier = Modifier.padding(start = 16.dp)) {
                        TagRows(activity, tagsViewModel, viewModel.tagIds, setTagIds = { viewModel.tagIds = it }, setShowTagsDialog = { showTagsDialog = it })
                    }
                }
            Row(Modifier.fillMaxWidth()) {
                // Fluid
                TextField(
                    value = viewModel.value1.value,
                    colors = textFieldColors,
                    onValueChange = {
                        viewModel.value1.value = it // Single source of truth!
                        if (shouldMoveFocus(it, viewModel.value1.template)) focusManager.moveFocus(FocusDirection.Next)
                        if (matchesTemplate(it, viewModel.value1.template)) viewModel.value1.value = it // Single source of truth!
                    },
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                    singleLine = true,
                    textStyle = TextStyle(),
                    label = { Text(text = "${viewModel.value1.label}*", maxLines = 1, overflow = TextOverflow.Ellipsis) },
                    placeholder = { Text("${viewModel.value1.template} ${viewModel.value1.unit}") },
                    modifier = Modifier
                        .weight(1f)
                        .padding(end = 20.dp)
                        .focusRequester(textField1),
                )
            }
            Row {
                // Comment
                TextField(
                    modifier = Modifier
                        .fillMaxWidth()
                        .focusRequester(textField2),
                    value = viewModel.comment.value,
                    colors = textFieldColors,
                    onValueChange = { viewModel.comment.value = it
                        //  commentString = it
                    },
                    singleLine = false,
                    label = { Text(text = viewModel.comment.label) },
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Text, capitalization = KeyboardCapitalization.Sentences),
                    trailingIcon = {
                        IconButton(onClick = {
                            showTextTemplatesDialog = true
                        })
                        { Icon(Icons.Outlined.Abc, contentDescription = null) }
                    },
                )
            }
            Text("")

            if (attachmentPresent()) {
                Row(modifier = Modifier.align(Alignment.CenterHorizontally)) {
                    AttachmentBlock()
                }
            }
        }

        // Dialogs
        if (showTagsDialog)
            tagsDialog.ShowDialog(viewModel.dataType, selectedTags = viewModel.tagIds, setShowDialog = { showTagsDialog = it }) {
                viewModel.tagIds = it
            }
        if (showTextTemplatesDialog)
            textTemplateDialog.ShowDialog(viewModel.dataType, setShowDialog = { showTextTemplatesDialog = it }) { viewModel.comment.value += it }

        if (showDatePickerDialog) OpenDatePickerDialog()
        if (showTimePickerDialog) OpenTimePickerDialog()    }


    //Tab specific pre-save checks
    @SuppressLint("SimpleDateFormat")
    override fun saveItem() {
        // Close keyboard so error is visible
        hideKeyboard()

        val validInput = matchesTemplate(viewModel.value1.value, viewModel.value1.template)
        if (!validInput) {
            val message = getString(string.invalidFormat) + "\n" + getString(string.configuredFormat, viewModel.value1.template)
            snackbarDelegate.showSnackbar(message)
            return
        }

        // Update Day field in case date got changed
        viewModel.value2.value = SimpleDateFormat(DAY_FORMAT).format(viewModel.timestamp)

        super.saveItem()
        finish() // Close current window / activity
    }

}