/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.documents

import android.content.Context
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.PlatformTextStyle
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.zell_mbc.medilog.MainActivity
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.base.BaseTab
import com.zell_mbc.medilog.data.Data
import com.zell_mbc.medilog.scaffold.Screens
import com.zell_mbc.medilog.support.SnackbarDelegate
import com.zell_mbc.medilog.tags.TagsViewModel

class DocumentsTab(override val label: String?, override val viewModel: DocumentsViewModel, override val tagsViewModel: TagsViewModel, context: Context, snackbarDelegate: SnackbarDelegate): BaseTab(viewModel, tagsViewModel, context, snackbarDelegate) {
    override var editActivityClass: Class<*> = DocumentsEditActivity::class.java
    override var infoActivityClass: Class<*> = DocumentsInfoActivity::class.java
    override var chartActivityClass: Class<*> = DocumentsChartActivity::class.java

    override val route = Screens.Documents.route
    override val fabEnabled = false
    override val iconActive = R.drawable.baseline_attach_file_24
    override val iconInactive = R.drawable.baseline_attach_file_24

    @Composable
    override fun ShowContent(padding: PaddingValues) {
        super.ShowContent(padding)

        LazyColumn(state = listState, horizontalAlignment = Alignment.Start, modifier = Modifier.fillMaxWidth().padding(start = 8.dp, end = 8.dp)) {
            items(dataList.value, key = { it._id }) { item ->
                HorizontalDivider(thickness = 1.dp, color = MaterialTheme.colorScheme.secondaryContainer) // Lines starting from the top
                Row(
                    modifier = Modifier.height(IntrinsicSize.Min).fillMaxWidth().width(IntrinsicSize.Max).clickable { selection.value = item },
                    verticalAlignment = Alignment.CenterVertically
                )
                {
                    ShowRow(item)
                }
                if (selection.value != null) {
                    ItemClicked(selection.value!!._id)
                    selection.value = null
                }
            }
        }
    }

    @Composable
    fun ShowRow(item: Data) {
        // Timestamp
        ResizableDateColumn(formatDateString(item.timestamp))
        Text(item.comment, Modifier.padding(start = cellPadding.dp, top = rowPadding.dp, bottom = rowPadding.dp).clickable {
                selection.value = item
                selectedField = "comment"
            },
            fontSize = MainActivity.fontSize.sp, style = TextStyle(lineHeight = TextUnit.Unspecified, platformStyle = PlatformTextStyle(includeFontPadding = false)) // style is for removing the padding between multiline text
        )
        if (item.attachment.isNotEmpty()) HandleAttachment(item.attachment)
    }

    override fun addItem() {
        startEditing(-1) // Indicate that this is about a new item
    }
}