/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.dialogs

import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.selection.toggleable
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.dialogs.Tabs.BLOODPRESSURE
import com.zell_mbc.medilog.dialogs.Tabs.DIARY
import com.zell_mbc.medilog.dialogs.Tabs.DOCUMENTS
import com.zell_mbc.medilog.dialogs.Tabs.FLUID
import com.zell_mbc.medilog.dialogs.Tabs.GLUCOSE
import com.zell_mbc.medilog.dialogs.Tabs.OXIMETRY
import com.zell_mbc.medilog.dialogs.Tabs.TEMPERATURE
import com.zell_mbc.medilog.dialogs.Tabs.WEIGHT
import com.zell_mbc.medilog.support.getAllTabMeta

// Tab definitions
object Tabs {
    const val WEIGHT = 1
    const val BLOODPRESSURE = 2
    const val DIARY = 3
    const val FLUID = 4
    const val GLUCOSE = 5
    const val TEMPERATURE = 6
    const val OXIMETRY = 7
    const val DOCUMENTS = 8
}

@Composable
fun TabsSelectionDialog(
    showDialog: Boolean,
    onDismiss: () -> Unit,
    onConfirm: (List<Int>) -> Unit
) {
    val context = LocalContext.current
    if (showDialog) {
        val allTabMeta = getAllTabMeta(context)
        val preselectedTabs = setOf(WEIGHT, BLOODPRESSURE, DIARY, DOCUMENTS)
        val (selected, setSelected) = remember(allTabMeta) {
            mutableStateOf(allTabMeta.map { (tabConst, _) -> preselectedTabs.contains(tabConst) })
        }

        AlertDialog(
            onDismissRequest = onDismiss,
            title = { Text(stringResource(R.string.tabSelectorDialogHeader)) },
            text = {
                Column(
                    modifier = Modifier.verticalScroll(rememberScrollState())
                ) {
                    Text(stringResource(R.string.tabSelectorDialogSubHeader))
                    Spacer(Modifier.height(8.dp))
                    HorizontalDivider(thickness = 1.dp)

                    allTabMeta.entries.forEachIndexed { index, (key, value) ->
                        Row(
                            verticalAlignment = Alignment.CenterVertically,
                            modifier = Modifier
                                .fillMaxWidth()
                                .toggleable(
                                    value = selected[index],
                                    onValueChange = {
                                        if (!selected[index] || selected.count { it } > 1) {
                                            setSelected(selected.toMutableList().also { it[index] = !it[index] })
                                        }
                                    }
                                )
                                .padding(vertical = 4.dp)
                        ) {
                            Checkbox(checked = selected[index], onCheckedChange = null)
                            Spacer(Modifier.width(8.dp))
                            Text(value)
                        }
                    }

                    HorizontalDivider(thickness = 1.dp)
                    Spacer(Modifier.height(8.dp))
                    Text(stringResource(R.string.tabSelectorDialogFooter))
                }
            },
            confirmButton = {
                Button(
                    enabled = selected.any { it },
                    onClick = {
                        val selectedTabs = allTabMeta.keys
                            .filterIndexed { idx, _ -> selected[idx] }
                        onConfirm(selectedTabs)
                    }
                ) {
                    Text(stringResource(R.string.ok))
                }
            },
            dismissButton = {
                OutlinedButton(onClick = onDismiss) {
                    Text(stringResource(R.string.cancel))
                }
            }
        )
    }
}
