/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.dialogs

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLinkStyles
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.withLink
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.window.DialogProperties
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.R

@Composable
fun ChangelogDialog(
    onDismissRequest: () -> Unit,
    modifier: Modifier = Modifier
) {
    val context = LocalContext.current
    val preferences = remember { PreferenceManager.getDefaultSharedPreferences(context) }
    val changelog = remember { context.resources.openRawResource(R.raw.changelog).bufferedReader().use { it.readText() } }

    // Load user text size from preferences (with fallback)
    var defaultFontSize = context.getString(R.string.TEXT_SIZE_DEFAULT).toInt()
    val checkValue = preferences.getString(SettingsActivity.KEY_PREF_TEXT_SIZE, null)
    if (!checkValue.isNullOrEmpty()) {
        defaultFontSize = checkValue.toIntOrNull() ?: defaultFontSize
    }

    AlertDialog(
        modifier = modifier.fillMaxWidth(),
        onDismissRequest = onDismissRequest,
        title = { Text("ChangeLog", fontSize = (defaultFontSize + 4).sp, fontWeight = FontWeight.Bold, color = MaterialTheme.colorScheme.primary) },
        text = { ChangelogContent(changelog = changelog, defaultFontSize = defaultFontSize) },
        confirmButton = { Button(onClick = onDismissRequest) { Text(stringResource(R.string.ok)) } },
        properties = DialogProperties(dismissOnBackPress = true, dismissOnClickOutside = true)
    )
}

@Composable
private fun ChangelogContent(changelog: String, defaultFontSize: Int) {
    val scrollState = rememberScrollState()
    var fw by remember { mutableStateOf(FontWeight.Normal) }
    var fs by remember { mutableStateOf(defaultFontSize) }
    var skipNextBlank = false

    Column(modifier = Modifier.verticalScroll(scrollState).padding(8.dp), horizontalAlignment = Alignment.Start) {
        for (line in changelog.lines()) {
            // Skip the "Changelog" header line entirely
            if (line.startsWith("# ")) {
                skipNextBlank = true
                continue
            }
            // If previous line was header, skip the following blank line
            if (skipNextBlank && line.isBlank()) {
                skipNextBlank = false
                continue
            } else skipNextBlank = false

            val text = when {
                line.startsWith("###") -> {
                    fw = FontWeight.Bold; fs = defaultFontSize; line.removePrefix("###").trim()
                }
                line.startsWith("##") -> {
                    fw = FontWeight.Bold; fs = defaultFontSize + 2; line.removePrefix("##").trim()
                }
                else -> {
                    fw = FontWeight.Normal; fs = defaultFontSize; line
                }
            }
            ShowRow(line = text, fs = fs, fw = fw)
        }
    }
}

@Composable
private fun ShowRow(line: String, fs: Int, fw: FontWeight) {
    val linkStart = line.indexOf("https://")
    val lineHeight = fs + 4

    if (linkStart < 0) {
        Text(line, color = MaterialTheme.colorScheme.primary, fontSize = fs.sp, lineHeight = lineHeight.sp, fontWeight = fw)
    } else {
        val prefix = line.substring(0, linkStart)
        val link = line.substring(linkStart)
        Column {
            if (prefix.isNotBlank()) {
                Text(prefix,
                    color = MaterialTheme.colorScheme.primary, fontSize = fs.sp,
                    lineHeight = lineHeight.sp,
                    fontWeight = fw
                )
            }
            Text(
                buildAnnotatedString {
                    withLink(
                        LinkAnnotation.Url(
                            url = link,
                            TextLinkStyles(style = SpanStyle(color = MaterialTheme.colorScheme.primary))
                        )
                    ) { append(link) }
                },
                textDecoration = TextDecoration.Underline,
                fontSize = fs.sp
            )
        }
    }
}
