/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.debug

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalConfiguration
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.LocalWindowInfo
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import kotlinx.coroutines.flow.MutableStateFlow

object DebugLog {
    // Debug mode state
    val debugMode = MutableStateFlow(false)
    fun toggleDebug(): Boolean {
        debugMode.value = !debugMode.value
        return debugMode.value
    }

    // Observable debug log for Compose
    val messages = mutableStateListOf("Debug log:")

    fun add(newLine: String) {
        // Inserts at index 1 (below the title), don't bother saving if debug mode is off
        if (debugMode.value) messages.add(1, newLine)
    }

    fun clear() {
        messages.clear()
        messages.add("Debug log:")
    }
}


@Composable
fun DebugOverlay(content: @Composable () -> Unit) {
    val windowInfo = LocalWindowInfo.current
    val density = LocalDensity.current
    val screenWidth = with(density) { windowInfo.containerSize.width.toDp() }
    val overlayWidth = screenWidth - 120.dp

    Box {
        content()
        // Debug log overlay
        LazyColumn(modifier = Modifier.width(overlayWidth).fillMaxHeight().background(Color(0x80000000)).padding(8.dp).align(Alignment.TopStart), horizontalAlignment = Alignment.Start) {
            items(DebugLog.messages) { line ->
                Text(text = line, color = Color.Yellow, fontSize = 12.sp, fontFamily = FontFamily.Monospace)
            }
        }
    }
}
