/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.data

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.os.StatFs
import android.widget.Toast
import androidx.core.net.toUri
import androidx.documentfile.provider.DocumentFile
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.databaseName
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.support.LAST_APP_VERSION
import java.io.File
import java.io.FileInputStream
import java.io.IOException

// Todo: The below needs thorough testing!!!
//  How often should it run? Should it replace auto backup?
// Needs to run before db is opened!
fun copyDatabaseFileToBackupFolder(context: Context) {
    val preferences = PreferenceManager.getDefaultSharedPreferences(context)

    val uriString = preferences.getString(SettingsActivity.KEY_PREF_BACKUP_URI, "")
    if (!uriString.isNullOrEmpty()) {
        val uri = uriString.toUri()
        val dFolder = DocumentFile.fromTreeUri(context, uri)
        if (dFolder != null && dFolder.canWrite()) {
            val backupFileName = "MediLogDatabase.backup" // e.g. "backup.zip"
            // Delete existing file with same name if any
            dFolder.findFile(backupFileName)?.delete()

            val dFile = dFolder.createFile("", backupFileName)
            if (dFile != null) {
                // Path to your app database file
                val sourceFile = File(context.getDatabasePath(databaseName).path) // or exact db file path
                if (sourceFile.exists()) {
                    try {
                        context.contentResolver.openOutputStream(dFile.uri)?.use { outputStream ->
                            FileInputStream(sourceFile).use { inputStream ->
                                inputStream.copyTo(outputStream)
                            }
                        } ?: run {
                            // Silent fail?
                            Handler(Looper.getMainLooper()).post {
                                Toast.makeText(context, context.getString(R.string.eUnableToWriteToFolder) + ": " + dFolder.uri, Toast.LENGTH_LONG).show()
                            }
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
            }
        }
    }

}

fun copyDatabaseToInternalFolder(context: Context) {
    val preferences = PreferenceManager.getDefaultSharedPreferences(context)
    val dbFile = context.getDatabasePath(databaseName)
    try {
        if (hasEnoughSpace(dbFile.parentFile, dbFile.length())) {
            val filesDir = context.filesDir
            val lastVersionCode = preferences.getInt(LAST_APP_VERSION, -1)
            val backupFile = File(filesDir, dbFile.name + "-$lastVersionCode.backup")

            // Delete other backup files with the same prefix but different version suffix
            filesDir.listFiles()?.forEach { file ->
                // Check for the pattern: dbname-<number>.backup but exclude the current version file
                if (file.name.startsWith(dbFile.name + "-") &&
                    file.name.endsWith(".backup") &&
                    file.name != backupFile.name) {
                    file.delete()
                }
            }
            dbFile.copyTo(backupFile, overwrite = true)
        } else Toast.makeText(context, "Insufficient diskspace, unable to create database backup copy", Toast.LENGTH_LONG).show()
    } catch (_: IOException) {
        // Handle copy failure, possibly log for support
    }
}

fun hasEnoughSpace(targetDir: File, backupSize: Long): Boolean {
    val stat = StatFs(targetDir.path)
    val available = stat.availableBytes
    return available > backupSize
}
