/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.data

import android.annotation.SuppressLint
import android.app.Application
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Handler
import android.os.Looper
import android.util.Log
import android.widget.Toast
import androidx.documentfile.provider.DocumentFile
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.MainActivity.Companion.DatePattern
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.support.*
import net.lingala.zip4j.io.outputstream.ZipOutputStream
import net.lingala.zip4j.model.ZipParameters
import net.lingala.zip4j.model.enums.EncryptionMethod
import java.io.BufferedOutputStream
import java.io.File
import java.io.OutputStream
import java.text.SimpleDateFormat
import java.util.*
import androidx.core.content.edit
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.ViewModelStoreOwner
import com.zell_mbc.medilog.backupFileNames
import com.zell_mbc.medilog.profiles.ProfilesViewModel
import com.zell_mbc.medilog.settings.SettingsViewModel
import com.zell_mbc.medilog.tags.TagsViewModel
import com.zell_mbc.medilog.texttemplates.TextTemplatesViewModel
import com.zell_mbc.medilog.weight.WeightViewModel

class Backup(storeOwner: ViewModelStoreOwner, val app: Context, private val uri: Uri, private val zipPassword: String) {
    //private val app = application
    val application = app.applicationContext as Application

    val viewModel = ViewModelProvider(storeOwner)[WeightViewModel::class.java]
    val profilesViewModel = ViewModelProvider(storeOwner)[ProfilesViewModel::class.java]

    val textTemplatesViewModel = ViewModelProvider(storeOwner)[TextTemplatesViewModel::class.java]
    val tagsViewModel= ViewModelProvider(storeOwner)[TagsViewModel::class.java]

    private val preferences = PreferenceManager.getDefaultSharedPreferences(app)
    private val datePrefix = preferences.getBoolean(SettingsActivity.KEY_PREF_ADD_TIMESTAMP, app.getString(R.string.ADD_TIMESTAMP_DEFAULT).toBoolean())

    // Create,write and protect backup ZIP file to backup location
    @SuppressLint("SimpleDateFormat")
    fun exportZIPFile(autoBackup: Boolean) {
        // persist permission to access backup folder across reboots
        val takeFlags = Intent.FLAG_GRANT_READ_URI_PERMISSION or Intent.FLAG_GRANT_WRITE_URI_PERMISSION
        try {
            app.grantUriPermission(app.packageName, uri, takeFlags)
            app.contentResolver.takePersistableUriPermission(uri, takeFlags)
        } catch (_: SecurityException) {
            Handler(Looper.getMainLooper()).post { Toast.makeText(app,app.getString(R.string.eLostAccessRights), Toast.LENGTH_LONG).show() }
            return
        }

        // Create empty Zip file
        var zipFile = app.getString(R.string.appName) + "-Backup" + if (datePrefix) "-" + SimpleDateFormat(DatePattern.DATE_TIME).format(Date().time) else ""
        zipFile = zipFile.replace(":","-") // German time format prevents a valid filename
        zipFile += ".zip"
        var dFile: DocumentFile? = null
        var dFolder: DocumentFile? = null
        try {
            dFolder = DocumentFile.fromTreeUri(app, uri)
        } catch (e: SecurityException) {
            Handler(Looper.getMainLooper()).post { Toast.makeText(app,app.getString(R.string.eCreateFile) + ": (1)" + dFile + " " + e.toString(), Toast.LENGTH_LONG).show() }
            return
        }
        if (dFolder == null) {
            Handler(Looper.getMainLooper()).post { Toast.makeText(app,app.getString(R.string.eCreateFile) + ": (2)" + dFile, Toast.LENGTH_LONG).show() }
            return
        }

        if (!dFolder.canWrite()) { // lost access rights, needs a manual backup
            Handler(Looper.getMainLooper()).post { Toast.makeText(app,app.getString(R.string.eLostAccessRights) + ": " + dFolder.toString(), Toast.LENGTH_LONG).show() }
            return
        }

        try {
            // Check if file exists, delete if yes
            if (dFolder.findFile(zipFile) != null) dFolder.findFile(zipFile)?.delete()
            dFile = dFolder.createFile("application/zip", zipFile)
            if (dFile == null) {
                Handler(Looper.getMainLooper()).post { Toast.makeText(app,app.getString(R.string.eUnableToWriteToFolder) + ": " + dFolder.toString(), Toast.LENGTH_LONG).show() }
                return
            }
        } catch (e: IllegalArgumentException) {
            Handler(Looper.getMainLooper()).post { Toast.makeText(app,app.getString(R.string.eCreateFile) + ": (3)" + dFile + " " + e.toString(), Toast.LENGTH_LONG).show() }
            return
        }

        // Add files to ZIP file
        val dest: OutputStream?
        val out: ZipOutputStream?

        try {
            dest = app.contentResolver.openOutputStream(dFile.uri)
            if (dest == null) {
                Handler(Looper.getMainLooper()).post { Toast.makeText(app,app.getString(R.string.eCreateFile) + " (4) " + dFile, Toast.LENGTH_LONG).show() }
                return
            }
            out = if (zipPassword.isEmpty()) ZipOutputStream(BufferedOutputStream(dest))
            else ZipOutputStream(BufferedOutputStream(dest), zipPassword.toCharArray())

        } catch (_: Exception) {
            Handler(Looper.getMainLooper()).post { Toast.makeText(app,app.getString(R.string.eCreateFile) + " (5) " + dFile, Toast.LENGTH_LONG).show() }
            return
        }
        val zipParameters = ZipParameters()
        if (zipPassword.isNotEmpty()) {
            zipParameters.isEncryptFiles = true
            zipParameters.encryptionMethod = EncryptionMethod.AES
        }

        // First we add the data file
        val dbTables = backupFileNames.count()
        var curFile = 0
        // Message get's in the way, think of better approach
        //Handler(Looper.getMainLooper()).post { Toast.makeText(app, app.getString(R.string.backupStarted), Toast.LENGTH_LONG).show() }
        val zipData = arrayOfNulls<String>(dbTables)
        zipData[curFile] = viewModel.dataToCSV(viewModel.backup())
        try {
            zipParameters.fileNameInZip = backupFileNames[curFile]
            out.putNextEntry(zipParameters)
            out.write(zipData[curFile]!!.toByteArray())
            out.closeEntry()
        } catch (_: Exception) {
            Handler(Looper.getMainLooper()).post { Toast.makeText(app, app.getString(R.string.eCreateFile) + " (6) " + zipParameters.fileNameInZip, Toast.LENGTH_LONG).show() }
            return
        }

        // Next the profiles table
        curFile++
        zipData[curFile] = profilesViewModel.dataToCSV(profilesViewModel.backup())
        try {
            zipParameters.fileNameInZip = backupFileNames[curFile]
            out.putNextEntry(zipParameters)
            out.write(zipData[curFile]!!.toByteArray())
            out.closeEntry()
        } catch (_: Exception) {
            Handler(Looper.getMainLooper()).post { Toast.makeText(app, app.getString(R.string.eCreateFile) + " (7) " + zipParameters.fileNameInZip, Toast.LENGTH_LONG).show() }
            return
        }

        // The text templates table
        curFile++
        if (textTemplatesViewModel.count() > 0) {
            zipData[curFile] = textTemplatesViewModel.dataToCSV(textTemplatesViewModel.backup())
            try {
                zipParameters.fileNameInZip = backupFileNames[curFile]
                out.putNextEntry(zipParameters)
                out.write(zipData[curFile]!!.toByteArray())
                out.closeEntry()
            } catch (_: Exception) {
                Handler(Looper.getMainLooper()).post { Toast.makeText(app, app.getString(R.string.eCreateFile) + " (8) " + zipParameters.fileNameInZip, Toast.LENGTH_LONG).show() }
                return
            }
        }

        // The Tags table
        curFile++
        if (tagsViewModel.count() > 0) {
            zipData[curFile] = tagsViewModel.dataToCSV(tagsViewModel.backup())
            try {
                zipParameters.fileNameInZip = backupFileNames[curFile]
                out.putNextEntry(zipParameters)
                out.write(zipData[curFile]!!.toByteArray())
                out.closeEntry()
            } catch (_: Exception) {
                Handler(Looper.getMainLooper()).post { Toast.makeText(app, app.getString(R.string.eCreateFile) + " (9) " + zipParameters.fileNameInZip, Toast.LENGTH_LONG).show() }
                return
            }
        }

        // The preference files
        val prefsDir = File(app.filesDir, "../shared_prefs")  // navigate to shared_prefs folder
        if (prefsDir.exists() && prefsDir.isDirectory) {
            val prefFiles = prefsDir.listFiles { file -> file.extension == "xml" } ?: emptyArray()
            var count = 0

            for (prefFile in prefFiles) {
                try {
                    zipParameters.fileNameInZip = prefFile.name
                    out.putNextEntry(zipParameters)
                    out.write(prefFile.readBytes())
                    out.closeEntry()
                    count++
                } catch (_: Exception) {
                    Handler(Looper.getMainLooper()).post { Toast.makeText(app, app.getString(R.string.eCreateFile) + " (11) ${prefFile.name}", Toast.LENGTH_LONG).show() }
                    return
                }
            }
            Log.i("Backup", "✅ Backed up $count preference files from ${prefsDir.path}")
        } else {
            Log.w("Backup", "⚠️ Preferences folder not found: ${prefsDir.path}")
        }

        // And now the attachments
        // Instead of saving all files in ./attachments build list from data.attachment. This allows to clean out orphans
        val sql = "SELECT * FROM data WHERE attachment <> ''"
        val liveAttachments = viewModel.getDataList(sql)

        // Compare attachmentsInFolder and liveAttachments size. If attachmentsInFolder > liveAttachments we got orphans, if it's the other way round we got a problem
        val attachmentsInFolder = listAttachments(app,  ATTACHMENT_LABEL)
        
        // Todo: Also shows during automated backups, need a more subtle way
        if (liveAttachments.count() != attachmentsInFolder.count()) {
            val s = "Error: Attachments in folder=${liveAttachments.count()}, db records=${attachmentsInFolder.count()}"
            //Handler(Looper.getMainLooper()).post { Toast.makeText(app,s, Toast.LENGTH_LONG).show() }
            Log.d("Attachments: ", s)
        }

        if (liveAttachments.isNotEmpty()) {
            val attachmentFolder = getAttachmentFolder(app)
            var i = 0
            for (item in liveAttachments) {
                val fileName = item.attachment
                try {
                    zipParameters.fileNameInZip = fileName
                    out.putNextEntry(zipParameters)

                    //val f = decryptAttachment(app, fileName)
                    val filePath = File(attachmentFolder, "")
                    val f = File(filePath, fileName)
                    if (f.exists()) {
                        out.write(f.readBytes())
                        out.closeEntry()
                        i++
                    }
                } catch (_: Exception) {
                    Handler(Looper.getMainLooper()).post { Toast.makeText(app, app.getString(R.string.eCreateFile) + " (11) " + fileName, Toast.LENGTH_LONG).show() }
                    return
                }
            }
        }

        // CLose ZIP file
        try {
            out.close()
        } catch (_: Exception) {
            Handler(Looper.getMainLooper()).post { Toast.makeText(app, app.getString(R.string.eCreateFile) + " (12) " + dFile, Toast.LENGTH_LONG).show() }
            return
        }
        val msg = if (zipPassword.isEmpty()) app.getString(R.string.unprotectedZipFileCreated)
        else app.getString(R.string.protectedZipFileCreated)

        val size = viewModel.backup().size
        val sizeString = "\n($size " + app.getString(R.string.entries) + ")"

        // No message if autoBackup to not annoy users
        if (!autoBackup) Handler(Looper.getMainLooper()).post { Toast.makeText(app, msg + " " + dFile.name + sizeString, Toast.LENGTH_LONG).show() }
        //if (autoBackup) msg = "AutoBackup: $msg"

        // Capture last backup date
        preferences.edit {
            putLong("LAST_BACKUP", Date().time)
            putString(SettingsActivity.KEY_PREF_BACKUP_URI, uri.toString())
        }
    }
}
