/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

/*
 */

package com.zell_mbc.medilog.bloodpressure

import android.os.Bundle
import androidx.activity.compose.setContent
import androidx.activity.viewModels
import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Abc
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.TextFieldColors
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusDirection
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.input.KeyboardCapitalization
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import kotlin.getValue
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.R.string
import com.zell_mbc.medilog.Tabs.BLOODPRESSURE
import com.zell_mbc.medilog.Thresholds
import com.zell_mbc.medilog.base.EditActivity
import com.zell_mbc.medilog.bloodpressure.MultipleMeasurementsDialog
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.support.checkMinMax
import com.zell_mbc.medilog.support.matchesTemplate
import com.zell_mbc.medilog.support.shouldMoveFocus

class BloodPressureEditActivity: EditActivity() {
    override val viewModel: BloodPressureViewModel by viewModels()

    override val dataType = BLOODPRESSURE

    private var heartRhythmIssue = false
    private var logHeartRhythm = false

    private var colorBad = androidx.compose.ui.graphics.Color.Black
    private var colorGood = androidx.compose.ui.graphics.Color.Black

    lateinit var multipleMeasurementsDialog: MultipleMeasurementsDialog
    var openMultipleMeasurementsDialog = mutableStateOf(false)

    override fun onCreate(savedInstanceState: Bundle?) {
        //viewModel = vm
        super.onCreate(savedInstanceState)

        logHeartRhythm = preferences.getBoolean(SettingsActivity.KEY_PREF_LOG_HEART_RHYTHM, true)

        if (logHeartRhythm) {
            //standardColors =  binding.etSys.currentHintTextColor //  textColors.defaultColor

            // Invert result because setHeartRhythm inverst again
//            binding.btHeartRhythm.setText(warning) // = (editItem.value4 != "1")
            heartRhythmIssue = (editItem.value4 != "1")
        }

        setContent { StartCompose() }
    }

    @Composable
    override fun ShowContent() {
        Column(modifier = Modifier.padding(start = 8.dp, end = 8.dp)) {
            DateTimeBlock()

            colorBad = MaterialTheme.colorScheme.error
            colorGood = MaterialTheme.colorScheme.primary
            var iconColor by remember { mutableStateOf(setIcon()) }

            val textFieldColors: TextFieldColors = TextFieldDefaults.colors(unfocusedContainerColor = androidx.compose.ui.graphics.Color.Transparent, focusedContainerColor = androidx.compose.ui.graphics.Color.Transparent)
            val (textField1, textField2, textField3) = remember { FocusRequester.createRefs() }

            Row(Modifier.fillMaxWidth()) {
                // Weight
                TextField(
                    value = viewModel.value1.value,
                    colors = textFieldColors,
                    onValueChange = {
                        // 1. Check template
                        if (matchesTemplate(it, viewModel.value1.template))
                        // 2. Max/min reached?
                            if (checkMinMax(field = it, min = 0, max = Thresholds.MAX_SYS)) {
                                viewModel.value1.value = it // Single source of truth!
                                // 3. Jump if needed
                                if (shouldMoveFocus(it, viewModel.value1.template, Thresholds.MAX_SYS))
                                    focusManager?.moveFocus(FocusDirection.Next)
                            }
                    },
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                    singleLine = true,
                    textStyle = TextStyle(),
                    label = { Text(text = "${viewModel.value1.label}*", maxLines = 1, overflow = TextOverflow.Ellipsis) },
                    placeholder = { Text("${viewModel.value1.template} ${viewModel.value1.unit}") },
                    modifier = Modifier.weight(1f).padding(end = 20.dp).focusRequester(textField1),
                )

                // Diastolic
                TextField(
                    value = viewModel.value2.value,
                    onValueChange = {
                        // 1. Check template
                        if (matchesTemplate(it, viewModel.value2.template))
                            // 2. Max/min reached?
                            if (checkMinMax(field = it, min = 0, max = Thresholds.MAX_DIA)) {
                                viewModel.value2.value = it // Single source of truth!
                                // 3. Jump if needed
                                if (shouldMoveFocus(it, viewModel.value2.template,Thresholds.MAX_DIA))
                                    focusManager?.moveFocus(FocusDirection.Next)
                            }
                    },
                    modifier = Modifier.weight(1f).padding(end = 20.dp).focusRequester(textField2),
                    colors = textFieldColors,
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                    singleLine = true,
                    textStyle = TextStyle(),
                    label = { Text(text = "${viewModel.value2.label}*", maxLines = 1, overflow = TextOverflow.Ellipsis) },
                    placeholder = { Text("${viewModel.value2.template} ${viewModel.value2.unit}") }
                    )
                // Pulse
                TextField(
                    value = viewModel.value3.value,
                    onValueChange = {
                        // 1. Check template
                        if (matchesTemplate(it, viewModel.value3.template))
                        // 2. Max/min reached?
                            if (checkMinMax(field = it, min = 0, max = Thresholds.MAX_PULSE)) {
                                viewModel.value3.value = it // Single source of truth!
                                // 3. Jump if needed
                                if (shouldMoveFocus(it, viewModel.value3.template, Thresholds.MAX_PULSE))
                                    focusManager?.moveFocus(FocusDirection.Next)
                            }
                    },
                    modifier = Modifier.weight(1f).padding(end = 20.dp).focusRequester(textField3),
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                    singleLine = true,
                    textStyle = TextStyle(),
                    label = { Text(text = viewModel.value3.label, maxLines = 1, overflow = TextOverflow.Ellipsis) },
                    placeholder = { Text("${viewModel.value3.template} ${viewModel.value3.unit}") },
                    colors = textFieldColors,
                )

                if (logHeartRhythm) {
                    Image(painter = painterResource(R.drawable.ic_baseline_warning_24), contentDescription = "Heart Rhythm",
                        modifier = Modifier
                            .weight(0.5f)
                            .padding(start = 16.dp)
                            .align(Alignment.CenterVertically)
                            .clickable { iconColor = setIcon() },
                        colorFilter = ColorFilter.tint(iconColor),
                    )
                }
            }
            Row {
                // Comment
                TextField(
                    modifier = Modifier.fillMaxWidth().focusRequester(textField3),
                    value = viewModel.comment.value,
                    colors = textFieldColors,
                    onValueChange = { viewModel.comment.value = it
                        //  commentString = it
                    },
                    singleLine = false,
                    label = { Text(text = viewModel.comment.label) },
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Text, capitalization = KeyboardCapitalization.Sentences),
                    trailingIcon = { IconButton(onClick = { showTextTemplatesDialog = true }) { Icon(Icons.Outlined.Abc, contentDescription = null) } },
                )
            }
            Text("")

            Row(modifier = Modifier.align(Alignment.CenterHorizontally)) {
                if (editItem.attachment.isNotEmpty()) AttachmentBlock()
            }
        }

        // Dialogs
        if (showTextTemplatesDialog)
            //TextTemplatesDialog(activity, setShowDialog = { showTextTemplatesDialog = it }) { commentString += it } // TextField Content
            textTemplateDialog.ShowDialog(viewModel.dataType, setShowDialog = { showTextTemplatesDialog = it }) { viewModel.comment.value += it }

        if (showDatePickerDialog) OpenDatePickerDialog()
        if (showTimePickerDialog) OpenTimePickerDialog()
        if (openMultipleMeasurementsDialog.value) multipleMeasurementsDialog.ShowDialog(setShowDialog = {
            openMultipleMeasurementsDialog.value = it
            finish() // Close current window / activity
        })
    }

    //Tab specific checks
    override fun saveItem() {
        hideKeyboard()

        var validInput = matchesTemplate(viewModel.value1)
        if (!validInput) {
            val message = getString(string.invalidFormat) + "\n" + getString(string.configuredFormat, viewModel.value1.template)
            snackbarDelegate.showSnackbar(message)
            return
        }

        validInput = matchesTemplate(viewModel.value2)
        if (!validInput) {
            val message = getString(string.invalidFormat) + "\n" + getString(string.configuredFormat, viewModel.value1.template)
            snackbarDelegate.showSnackbar(message)
            return
        }
        if (!viewModel.value3.value.isEmpty()) {
            validInput = matchesTemplate(viewModel.value3)
            if (!validInput) {
                val message = getString(string.invalidFormat) + "\n" + getString(string.configuredFormat, viewModel.value1.template)
                snackbarDelegate.showSnackbar(message)
                return
            }
        }

        viewModel.value4.value = if (heartRhythmIssue) "1" else "0"

        super.saveItem()

        val multipleMeasurementsTimeWindow = "" + preferences.getString(SettingsActivity.KEY_PREF_BLOODPRESSURE_MULTI_MEASUREMENTS_WINDOW, getString(R.string.BLOODPRESSURE_MULTI_MEASUREMENTS_WINDOW_DEFAULT))
        if (multipleMeasurementsTimeWindow.toInt() > 0) {
            multipleMeasurementsDialog = MultipleMeasurementsDialog(this, viewModel as BloodPressureViewModel)
            openMultipleMeasurementsDialog.value = true
        }
        else finish() // Close current window / activity

    }

    private fun setIcon(): androidx.compose.ui.graphics.Color {
        heartRhythmIssue = !heartRhythmIssue
        return if (heartRhythmIssue) colorBad else colorGood
    }
}
