/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.about

import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Build
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.Launch
import androidx.compose.material.icons.filled.Backup
import androidx.compose.material.icons.filled.Clear
import androidx.compose.material.icons.filled.DiscFull
import androidx.compose.material.icons.filled.Launch
import androidx.compose.material.icons.filled.Lock
import androidx.compose.material.icons.filled.Memory
import androidx.compose.material.icons.filled.Numbers
import androidx.compose.material.icons.filled.Password
import androidx.compose.material.icons.filled.Person
import androidx.compose.material.icons.filled.PlayArrow
import androidx.compose.material.icons.filled.Screenshot
import androidx.compose.material.icons.filled.Start
import androidx.compose.material.icons.outlined.Email
import androidx.compose.material.icons.outlined.Web
import androidx.compose.material3.Button
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLinkStyles
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.withLink
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.unit.dp
import com.zell_mbc.medilog.AppFlavours
import com.zell_mbc.medilog.ProjectDetails
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.support.currentLocale
import com.zell_mbc.medilog.support.link
import java.io.ByteArrayInputStream
import java.io.InputStream
import java.security.cert.CertificateException
import java.security.cert.CertificateFactory
import java.security.cert.X509Certificate
import java.util.Calendar
import androidx.core.net.toUri
import com.zell_mbc.medilog.support.getCertificateSubject
import java.text.DateFormat

val good = 0xFF388E3C
val bad = 0xFFD32F2F

@Composable
fun SecuritySection(biometricLogonActive: Boolean, passwordProtectedBackups: Boolean, blockScreenShots: Boolean) {
    val context = LocalContext.current

Surface(modifier = Modifier.fillMaxWidth(), tonalElevation = 6.dp, shape = MaterialTheme.shapes.medium) {
    /*
        if (preferences.getBoolean("encryptedDB", false)) {
            s = getString(R.string.databaseEncryption) + ": " + if (preferences.getBoolean("encryptedDB", false)) getString(R.string.onSwitch)
            else getString(R.string.offSwitch)
            Text(s, style = MaterialTheme.typography.bodyMedium, color = MaterialTheme.colorScheme.primary)
        }

        @Suppress("DEPRECATION") val sig =
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) context.packageManager.getPackageInfo(context.packageName, PackageManager.GET_SIGNING_CERTIFICATES).signingInfo?.apkContentsSigners[0]
            else context.packageManager.getPackageInfo(context.packageName, PackageManager.GET_SIGNATURES).signatures?.get(0)

        val certStream: InputStream = ByteArrayInputStream(sig?.toByteArray())
        val subject = try {
            val certFactory = CertificateFactory.getInstance("X509")
            val x509Cert: X509Certificate = certFactory.generateCertificate(certStream) as X509Certificate
            x509Cert.subjectDN.toString()
        } catch (_: CertificateException) {
            "Unknown"
        }*/

    val subject = getCertificateSubject(context)

    Column(modifier = Modifier.padding(16.dp)) {
            Text(text = stringResource(id = R.string.security), style = MaterialTheme.typography.titleMedium)
            //Text(stringResource(id = R.string.security), modifier = Modifier.Companion.padding(top = 8.dp), style = MaterialTheme.typography.bodyLarge, color = MaterialTheme.colorScheme.primary)
            Spacer(Modifier.height(4.dp))
            SecurityRow(
                icon = if (biometricLogonActive) Icons.Default.Lock else Icons.Default.Clear,
                iconTint = if (biometricLogonActive) Color(good) else Color(bad),
                label = stringResource(R.string.biometricProtection),
                value = if (biometricLogonActive) stringResource(R.string.onSwitch) else stringResource(R.string.offSwitch)
            )
            SecurityRow(
                icon = if (passwordProtectedBackups) Icons.Default.Password else Icons.Default.Clear,
                iconTint = if (passwordProtectedBackups) Color(good) else Color(bad),
                label = stringResource(R.string.protectedBackup),
                value = if (passwordProtectedBackups) stringResource(R.string.onSwitch) else stringResource(R.string.offSwitch)
            )
            SecurityRow(
                icon = if (blockScreenShots) Icons.Default.Screenshot else Icons.Default.Clear,
                iconTint = if (blockScreenShots) Color(good) else Color(bad),
                label = stringResource(R.string.blockScreenshots),
                value = if (blockScreenShots) stringResource(R.string.onSwitch) else stringResource(R.string.offSwitch)
            )
            InfoRow(label = stringResource(R.string.appCertificate) + ": ", value = subject)
        }
    }
}


@Composable
fun StatisticSection(unlimitedRecords: Boolean, recordCount: Int, flavour: Int, databaseSize: String, attachments: Int, attachmentSize: String, profileCount: String, memoryConsumption: String, timestamp: Long, onBackupClick: () -> Unit) {
    Surface(modifier = Modifier.fillMaxWidth(), tonalElevation = 6.dp, shape = MaterialTheme.shapes.medium) {
        Column(modifier = Modifier.padding(16.dp), verticalArrangement = Arrangement.spacedBy(12.dp)
        ) {
            Text(text = stringResource(id = R.string.statistics), style = MaterialTheme.typography.titleMedium, color = MaterialTheme.colorScheme.primary)
        }
            val dataRecordCount = "(" + (if (unlimitedRecords) stringResource(R.string.unlimited) else AppFlavours.PLAY_NOT_LICENSED.toString()) + ") ${recordCount.toString()}"
            val textColor = if (flavour == AppFlavours.PLAY && recordCount >= AppFlavours.PLAY_NOT_LICENSED_WARNING) MaterialTheme.colorScheme.error else MaterialTheme.colorScheme.onSurfaceVariant

            val databaseOnDiskSize = stringResource(R.string.database) + " " + stringResource(R.string.size) + ": $databaseSize"

            Column(modifier = Modifier.padding(16.dp)) {
                Text(stringResource(R.string.statistics), style = MaterialTheme.typography.titleMedium)
                Spacer(Modifier.height(4.dp))
                StatisticRow(icon = Icons.Default.Memory, label = stringResource(R.string.memoryConsumption), value = memoryConsumption)
                StatisticRow(icon = Icons.Default.Person, label = stringResource(R.string.profiles), value = profileCount)
                StatisticRow(icon = Icons.Default.Numbers, label = stringResource(R.string.databaseEntries), value = dataRecordCount, color = textColor)
                StatisticRow(icon = Icons.Default.DiscFull, label = databaseOnDiskSize, value = databaseSize)
                //  if (attachments > 0) {
                StatisticRow(icon = Icons.Default.Numbers, label = stringResource(R.string.attachments), value = attachments.toString())
                StatisticRow(icon = Icons.Default.DiscFull, label = stringResource(R.string.size), value = attachmentSize)

                // ------------ Backup -----------
                val dateFormat = DateFormat.getDateInstance(DateFormat.SHORT)
                val timeFormat = DateFormat.getTimeInstance(DateFormat.SHORT)


                var icon = Icons.Default.Backup
                val lastBackup = if (timestamp > 0) " " + dateFormat.format(timestamp) + " " + timeFormat.format(timestamp)
                else {
                    icon = Icons.Default.Clear
                    stringResource(id = R.string.never)
                }
                Row(verticalAlignment = Alignment.Companion.CenterVertically) {
                    Icon(imageVector = icon, contentDescription = null, modifier = Modifier.size(20.dp))
                    Spacer(Modifier.width(10.dp))
                    Text(text = stringResource(R.string.lastBackup), style = MaterialTheme.typography.bodyMedium, modifier = Modifier.weight(1f))
                    Text(lastBackup, style = MaterialTheme.typography.bodyMedium)
                    IconButton(onClick = onBackupClick,  modifier = Modifier.size(36.dp) // Slightly smaller and cleaner than a full button
                    ) { Icon(imageVector = Icons.Default.PlayArrow, tint = Color(good), contentDescription = "Backup now") }
                }
            }
    }
}

@Composable
fun ContactSection() {
    val context = LocalContext.current
    Surface(modifier = Modifier.fillMaxWidth(), tonalElevation = 6.dp, shape = MaterialTheme.shapes.medium) {
        Column(modifier = Modifier.padding(16.dp), verticalArrangement = Arrangement.spacedBy(12.dp)) {
            val iconButtonModifier = Modifier.Companion.size(30.dp) // smaller size than default 48dp

            var url = ProjectDetails.authorUrl
            val currentLanguage = currentLocale(context)
            url += when (currentLanguage) {
                "de" -> "/de"
                else -> ""
            }
            Text(text = stringResource(id = R.string.contact), style = MaterialTheme.typography.titleMedium)
            Row(modifier = Modifier.padding(all = 0.dp)) {
                Text(stringResource(R.string.author) + ": ", style = MaterialTheme.typography.bodyMedium)
                Text(buildAnnotatedString {
                    //append(stringResource(R.string.author) +":")
                    withLink(LinkAnnotation.Url(url = url, TextLinkStyles(style = SpanStyle(color = MaterialTheme.colorScheme.link))))
                    { append(ProjectDetails.author) }
                }, style = MaterialTheme.typography.bodyMedium, textDecoration = TextDecoration.Underline)
            }

            Column(verticalArrangement = Arrangement.spacedBy(0.dp), modifier = Modifier.Companion.padding(0.dp)) {
                // Email Row
                Row(verticalAlignment = Alignment.Companion.CenterVertically, modifier = Modifier.Companion.height(IntrinsicSize.Min)) {
                    IconButton(
                        onClick = {
                            val intent = Intent(Intent.ACTION_VIEW, Uri.parse("mailto:" + ProjectDetails.emailAddress))
                            context.startActivity(intent)
                        },
                        modifier = iconButtonModifier
                    ) {
                        Icon(imageVector = Icons.Outlined.Email, contentDescription = stringResource(R.string.emailDescription), modifier = Modifier.Companion.size(24.dp)) // icon size inside button)
                    }
                    Text(" " + stringResource(R.string.emailDescription), style = MaterialTheme.typography.bodyMedium, modifier = Modifier.Companion.padding(vertical = 0.dp) // no extra vertical padding
                    )
                }

                // Fediverse Row
                Row(
                    verticalAlignment = Alignment.Companion.CenterVertically,
                    modifier = Modifier.Companion.height(IntrinsicSize.Min)
                ) {
                    IconButton(
                        onClick = {
                            val intent = Intent(Intent.ACTION_VIEW, Uri.parse(ProjectDetails.fediverseUrl))
                            context.startActivity(intent)
                        },
                        modifier = iconButtonModifier
                    ) {
                        Icon(
                            painterResource(R.drawable.fediverse_icon),
                            contentDescription = stringResource(R.string.fediverseDescription),
                            modifier = Modifier.Companion.size(24.dp)
                        )
                    }
                    Text(" " + stringResource(R.string.fediverseDescription), style = MaterialTheme.typography.bodyMedium, modifier = Modifier.Companion.padding(vertical = 0.dp))
                }

                // Website Row
                Row(
                    verticalAlignment = Alignment.Companion.CenterVertically,
                    modifier = Modifier.Companion.height(IntrinsicSize.Min)
                ) {
                    IconButton(
                        onClick = {
                            val intent = Intent(Intent.ACTION_VIEW, Uri.parse(ProjectDetails.projectWebsiteUrl))
                            context.startActivity(intent)
                        },
                        modifier = iconButtonModifier
                    ) {
                        Icon(imageVector = Icons.Outlined.Web,
                            contentDescription = stringResource(R.string.websiteDescription),
                            modifier = Modifier.Companion.size(24.dp)
                        )
                    }
                    Text(
                        " " + stringResource(R.string.websiteDescription),
                        style = MaterialTheme.typography.bodyMedium,
                        modifier = Modifier.Companion.padding(vertical = 0.dp)
                    )
                }
            }
        }
    }
}

@Composable
fun ProjectInfoSection() {
    val context = LocalContext.current
    val year = Calendar.getInstance().get(Calendar.YEAR)
    Surface(
        modifier = Modifier.fillMaxWidth(),
        tonalElevation = 6.dp,
        shape = MaterialTheme.shapes.medium
    ) {
        Column(
            modifier = Modifier.padding(16.dp),
            verticalArrangement = Arrangement.spacedBy(8.dp)
        ) {
            Text(text = stringResource(R.string.project), style = MaterialTheme.typography.titleMedium)

            Text(
                buildAnnotatedString {
                    withStyle(SpanStyle(color = MaterialTheme.colorScheme.link, textDecoration = TextDecoration.Underline)) {
                        append(stringResource(id = R.string.thirdParty))
                    }
                },
                modifier = Modifier.clickable {
                    val browserIntent = Intent(Intent.ACTION_VIEW, ProjectDetails.thirdParty.toUri())
                    context.startActivity(browserIntent)
                },
                style = MaterialTheme.typography.bodySmall,
                color = MaterialTheme.colorScheme.link
            )

            Text(
                buildAnnotatedString {
                    withStyle(SpanStyle(color = MaterialTheme.colorScheme.link, textDecoration = TextDecoration.Underline)) {
                        append(stringResource(id = R.string.copyright, year))
                    }
                },
                modifier = Modifier.clickable {
                    // Handle open license URL here
                    val browserIntent = Intent(Intent.ACTION_VIEW, ProjectDetails.appLicense.toUri())
                    context.startActivity(browserIntent)
                },
                style = MaterialTheme.typography.bodySmall,
                color = MaterialTheme.colorScheme.link
            )
        }
    }
}


