/*
 *     This file is part of MediLog.
 *
 *     MediLog is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Affero General Public License as published by
 *     the Free Software Foundation.
 *
 *     MediLog is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Affero General Public License for more details.
 *
 *     You should have received a copy of the GNU Affero General Public License
 *     along with MediLog.  If not, see <http://www.gnu.org/licenses/>.
 *
 *     Copyright (c) 2018 - 2025 by Zell-MBC.com
 */

package com.zell_mbc.medilog.about

import android.app.Activity
import android.content.Context
import android.content.ContextWrapper
import androidx.compose.runtime.State as ComposeState
import android.content.SharedPreferences
import android.os.Handler
import android.os.Looper
import android.text.format.Formatter
import android.widget.Toast
import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Info
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableLongStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLinkStyles
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.withLink
import androidx.compose.ui.unit.dp
import androidx.core.net.toUri
import androidx.documentfile.provider.DocumentFile
import androidx.lifecycle.ViewModelStoreOwner
import androidx.lifecycle.viewModelScope
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.preference.PreferenceManager
import com.zell_mbc.medilog.AppFlavours
import com.zell_mbc.medilog.R
import com.zell_mbc.medilog.UnlimitedRecords
import com.zell_mbc.medilog.data.Backup
import com.zell_mbc.medilog.debug.DebugLog
import com.zell_mbc.medilog.preferences.SettingsActivity
import com.zell_mbc.medilog.profiles.ProfilesViewModel
import com.zell_mbc.medilog.support.ATTACHMENT_LABEL
import com.zell_mbc.medilog.support.attachmentSize
import com.zell_mbc.medilog.support.countAttachments
import com.zell_mbc.medilog.support.getCertificateSubject
import com.zell_mbc.medilog.support.getMemoryConsumption
import com.zell_mbc.medilog.support.getVersionCode
import com.zell_mbc.medilog.support.getVersionName
import com.zell_mbc.medilog.support.link
import com.zell_mbc.medilog.weight.WeightViewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun AboutScreen(onBack: () -> Unit) {
        Scaffold(topBar = {
            TopAppBar(title = { Text(stringResource(R.string.action_about)) },
                    navigationIcon = { IconButton(onClick = onBack) {
                        Icon(Icons.AutoMirrored.Filled.ArrowBack, contentDescription = "Back")
                        }
                    }
                )
            }
        ) { innerPadding -> ShowContent(padding = innerPadding) }
}

@Composable
fun ShowContent(padding: PaddingValues = PaddingValues()) {
    val context = LocalContext.current

    val viewModel: WeightViewModel = viewModel()
    val profilesViewModel: ProfilesViewModel = viewModel()

    var databaseSize: String
    var blockScreenShots: Boolean
    var unlimitedRecords: Boolean

    val preferences = PreferenceManager.getDefaultSharedPreferences(context)
    unlimitedRecords = preferences.getBoolean(UnlimitedRecords,false)

    blockScreenShots = preferences.getBoolean(SettingsActivity.Companion.KEY_PREF_BLOCK_SCREENSHOTS, false)
    //if (blockScreenShots) window.setFlags(WindowManager.LayoutParams.FLAG_SECURE, WindowManager.LayoutParams.FLAG_SECURE)

    // Retrieve data
    val recordCount: Int = viewModel.count()
    val f = context.getDatabasePath("MediLogDatabase")
    databaseSize = Formatter.formatFileSize(context, f?.length() ?: 0)

    val memoryConsumption: String = Formatter.formatFileSize(context, getMemoryConsumption())
    val profileCount: String = profilesViewModel.count().toString()

    val l = attachmentSize(context, ATTACHMENT_LABEL)
    val attachmentSize = Formatter.formatFileSize(context, l)
    val attachments = countAttachments(context, ATTACHMENT_LABEL)

    val versionCode: String = getVersionCode(context) // Build number
    val flavour: Int = context.resources.getInteger(R.integer.flavour)
    val certificateSubject = getCertificateSubject(context)
    val versionName = if (flavour != AppFlavours.PLAY && certificateSubject.contains("zell-mbc.com")) {
        val url = if (context.packageName.contains(".fdroid")) "https://verification.f-droid.org/packages/com.zell_mbc.medilog.fdroid/" else "https://apt.izzysoft.de/fdroid/index/apk/com.zell_mbc.medilog"
        buildAnnotatedString {
            append(getVersionName(context) + ", " + context.packageName)
            withLink(LinkAnnotation.Url(url = url, TextLinkStyles(style = SpanStyle(color = MaterialTheme.colorScheme.link))))
            { append(" (" + stringResource(R.string.reproducible) + ")") }
        }
    }
    else AnnotatedString(getVersionName(context) + ", " + context.packageName)

    val scrollState = rememberScrollState()

    Column(modifier = Modifier.fillMaxSize().padding(padding).verticalScroll(scrollState).padding(horizontal = 16.dp, vertical = 12.dp), // your fixed content padding
            verticalArrangement = Arrangement.spacedBy(16.dp)) {
        Row {
            LogoWithTapSetting(onFiveTaps = {
                // This block runs after 5 taps
                val msg = if (DebugLog.toggleDebug()) "enabled" else "disabled"
                Toast.makeText(context, "Debug mode $msg", Toast.LENGTH_LONG).show()
            })
            Column(modifier = Modifier.align(alignment = Alignment.CenterVertically)) {
                Text(text = stringResource(id = R.string.appName), style = MaterialTheme.typography.headlineMedium, textAlign = TextAlign.Center)
                Spacer(modifier = Modifier.height(4.dp))
                Text(
                    text = stringResource(id = R.string.appDescription),
                    style = MaterialTheme.typography.bodyMedium,
                    color = MaterialTheme.colorScheme.onBackground,
                    textAlign = TextAlign.Center
                )
            }
        }

        Surface(
            modifier = Modifier.fillMaxWidth(),
            tonalElevation = 6.dp,
            shape = MaterialTheme.shapes.medium
        ) {
            Column(modifier = Modifier.padding(16.dp), verticalArrangement = Arrangement.spacedBy(8.dp)) {
                Text(text = stringResource(id = R.string.version), style = MaterialTheme.typography.titleMedium)
                Text(text = versionName, style = MaterialTheme.typography.bodyMedium)
                Text(text = "Build: $versionCode", style = MaterialTheme.typography.bodyMedium)
                if (flavour == AppFlavours.PLAY) {
                    val s = stringResource(id = R.string.license) + ": " + (if (unlimitedRecords) stringResource(R.string.unlimited) else stringResource(R.string.freeEdition))
                    Text(text = s, style = MaterialTheme.typography.bodyMedium)
                }
                //val certificateSubject = getCertificateSubject(context)
                /*
                const val FOSS = 0  // Standard build signed by me and used by F-Droid, Codeberg and Izzy -> Reproducibility validated by Izzy
                const val PLAY = 1 // Google Play only -> Reproducibility does not apply
                const val FDROID = 2 // F-Droid reproducible build, signed by me

                | Flavour | Signed by me?  | Expected warning | About dialog
                | ------- | -------------- | ---------------- |
                | FOSS    | yes            | No warning       | Izzy & Codeberg, etc.
                | FOSS    | no             | Show warning     | F-Droid signature, debug or copycat
                | PLAY    | no             | No warning       | Never show a warning, always signed by Google :-/
                | FDROID  | yes            | No warning       | New F-Droid com.zell_mbc.medilog.fdroid flavour
                | FDROID  | no             | Show warning     | Copycat
                */
                // Never show if PLAY, else show if not signed by me
                if (flavour != AppFlavours.PLAY && !certificateSubject.contains("zell-mbc.com")) {
                    // 🔥 Show reproducible build warning box
                    ElevatedCard(colors = CardDefaults.cardColors(containerColor = MaterialTheme.colorScheme.errorContainer),
                        modifier = Modifier.fillMaxWidth(), elevation = CardDefaults.elevatedCardElevation(defaultElevation = 4.dp)) {
                        Column(modifier = Modifier.padding(16.dp), verticalArrangement = Arrangement.spacedBy(8.dp)) {
                            Row(
                                verticalAlignment = Alignment.CenterVertically,
                                horizontalArrangement = Arrangement.spacedBy(8.dp)
                            ) {
                                Icon(imageVector = Icons.Default.Info, contentDescription = null, tint = MaterialTheme.colorScheme.error)
                                Text(text = stringResource(R.string.reproducibleTitle), style = MaterialTheme.typography.titleMedium, color = MaterialTheme.colorScheme.error)
                            }
                            Text(text = stringResource(id = R.string.notReproducible), style = MaterialTheme.typography.bodyMedium)
                            Text(buildAnnotatedString {
                                append(stringResource(R.string.reproducibleMessage) + " ")
                                withLink(LinkAnnotation.Url(url = stringResource(id = R.string.fdroidReproducibleLink), TextLinkStyles(style = SpanStyle(color = MaterialTheme.colorScheme.link))))
                                { append(stringResource(id = R.string.fdroidReproducibleLink)) }
                            }, style = MaterialTheme.typography.bodyMedium)
                            Text(buildAnnotatedString {
                                append(stringResource(R.string.reproducibleDetails) + " ")
                                withLink(LinkAnnotation.Url(url = stringResource(id = R.string.reproducibleDetailsLink), TextLinkStyles(style = SpanStyle(color = MaterialTheme.colorScheme.link))))
                                { append(stringResource(id = R.string.reproducibleDetailsLink)) }
                            }, style = MaterialTheme.typography.bodyMedium)
                        }
                    }
                }
            }
        }
        ContactSection()

        val timestampState = observeLastBackup(preferences)
        val timestamp = timestampState.value
        val emptyDb =  stringResource(R.string.emptyDatabase)
        StatisticSection(unlimitedRecords = unlimitedRecords, recordCount = recordCount, flavour = flavour,
            attachments = attachments, attachmentSize = attachmentSize, profileCount = profileCount, databaseSize = databaseSize, memoryConsumption = memoryConsumption, timestamp = timestamp,
            onBackupClick = {
                //Nothing to do
                if (recordCount == 0) {
                    Toast.makeText(context, emptyDb, Toast.LENGTH_LONG).show()
                }
                else viewModel.viewModelScope.launch(Dispatchers.IO) {
                    runBackup(context)
                }
            }
        )
        SecuritySection(
            preferences.getBoolean(SettingsActivity.Companion.KEY_PREF_BIOMETRIC, false),
            (preferences.getString(SettingsActivity.Companion.KEY_PREF_PASSWORD, ""))?.isNotEmpty() ?: false,
            blockScreenShots
        )
        ProjectInfoSection()
    }
}

@Composable
fun observeLastBackup(preferences: SharedPreferences): ComposeState<Long> {
    val state = remember { mutableLongStateOf(preferences.getLong("LAST_BACKUP", 0L)) }

    val listener = remember {
        SharedPreferences.OnSharedPreferenceChangeListener { prefs, key ->
            if (key == "LAST_BACKUP") {
                state.longValue = prefs.getLong(key, 0L)
            }
        }
    }

    DisposableEffect(preferences) {
        preferences.registerOnSharedPreferenceChangeListener(listener)
        onDispose {
            preferences.unregisterOnSharedPreferenceChangeListener(listener)
        }
    }

    return state
}

fun Context.findActivity(): Activity? {
    var context = this
    while (context is ContextWrapper) {
        if (context is Activity) return context
        context = context.baseContext
    }
    return null
}

@Composable
fun LogoWithTapSetting(
    modifier: Modifier = Modifier,
    onFiveTaps: () -> Unit
) {
    var tapCount by remember { mutableIntStateOf(0) }

    Image(
        painter = painterResource(id = R.mipmap.ic_launcher_inverted),
        contentDescription = stringResource(id = R.string.appName),
        modifier = modifier.size(100.dp).clickable {
            tapCount++
            if (tapCount == 5) {
                onFiveTaps()
                tapCount = 0
            }
        }
    )
}

fun runBackup(context: Context){
    val preferences = PreferenceManager.getDefaultSharedPreferences(context)

    // Check Uri
    val uriString = preferences.getString(SettingsActivity.KEY_PREF_BACKUP_URI, "")
    if (uriString.isNullOrEmpty()) {
        Handler(Looper.getMainLooper()).post { Toast.makeText(context, context.getString(R.string.missingBackupLocation), Toast.LENGTH_LONG).show() }
        false //Log.d("Preference", "i")
    } else {
        // Password
        var zipPassword = preferences.getString(SettingsActivity.KEY_PREF_PASSWORD, "")
        if (zipPassword.isNullOrEmpty()) zipPassword = ""
        //Log.d("Preference", "i")

        val uri = uriString.toUri()
        val dFolder = DocumentFile.fromTreeUri(context, uri)
        if (dFolder == null) { Handler(Looper.getMainLooper()).post { Toast.makeText(context, context.getString(R.string.missingBackupLocation), Toast.LENGTH_LONG).show() } }
        else {
            val activity = context.findActivity()
            Backup(activity!! as ViewModelStoreOwner,context, uri, zipPassword).exportZIPFile(false)
        }
        true
    }
}